# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # Feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to discover the location of the light switch
        # and ensure the robot can reach it.

        # Try to find the light switch location in the environment
        light_switch_location = None
        for key in positions:
            if 'light_switch' in key or 'switch' in key:
                light_switch_location = key
                break

        if light_switch_location is None:
            print("[Exploration] Could not find light switch location in object positions.")
            # Optionally, list all positions for debugging
            print("[Exploration] Available positions:", positions.keys())
            # End exploration if not found
            return

        print(f"[Exploration] Light switch location identified as: {light_switch_location}")

        # Now, check where the robot currently is
        # Try to find the robot's current location from the observation or positions
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = key
                break

        if robot_location is None:
            # Fallback: try to get from observation if available
            try:
                robot_location = obs['robot_location']
            except Exception:
                print("[Exploration] Could not determine robot's current location.")
                return

        print(f"[Exploration] Robot current location: {robot_location}")

        # If robot is not at the light switch, move it there using the available skill
        # Use 'execute_go' skill: execute_go(from_location, to_location)
        if robot_location != light_switch_location:
            print(f"[Exploration] Moving robot from {robot_location} to {light_switch_location} using execute_go.")
            try:
                obs, reward, done, info = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=light_switch_location
                )
                print(f"[Exploration] Robot moved to {light_switch_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")
                return
        else:
            print("[Exploration] Robot is already at the light switch location.")

        # After moving, check if the predicate (robot-at light_switch) is now true
        # This can be checked via the environment's state or observation
        # For demonstration, print out the current robot location again
        try:
            updated_positions = get_object_positions()
            updated_robot_location = None
            for key in updated_positions:
                if 'robot' in key or 'base' in key:
                    updated_robot_location = key
                    break
            if updated_robot_location == light_switch_location:
                print("[Exploration] Predicate (robot-at light_switch) is now true.")
            else:
                print("[Exploration] Predicate (robot-at light_switch) is still missing.")
        except Exception as e:
            print(f"[Exploration] Could not verify robot-at predicate: {e}")

        # === End of Exploration Phase ===

        # At this point, the missing predicate has been explored and the robot is at the light switch.
        # You can now proceed with the rest of the oracle plan as needed, using only the available skills.

        # Example: If the next step is to push the switch to turn on the light
        # Use 'execute_push_switch' if available, otherwise use the closest available skill
        if 'execute_push_switch' in globals():
            try:
                print("[Task] Executing execute_push_switch to turn on the light.")
                obs, reward, done, info = execute_push_switch(
                    env,
                    task,
                    switch=light_switch_location,
                    location=light_switch_location
                )
                print("[Task] Light should now be on (room-bright).")
            except Exception as e:
                print(f"[Task] Error during execute_push_switch: {e}")
        else:
            print("[Task] No execute_push_switch skill available; skipping light activation.")

        # Continue with further plan steps as required by the task...

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()