# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-at light_switch) ===
        # The feedback indicates that the predicate (robot-at light_switch) is missing.
        # We need to explore the environment to find the location of the light switch.

        # Try to find the light switch in the positions dictionary
        light_switch_pos = None
        for obj_name in positions:
            if "light_switch" in obj_name or "switch" in obj_name:
                light_switch_pos = positions[obj_name]
                light_switch_name = obj_name
                break

        if light_switch_pos is None:
            print("[Exploration] Light switch not found in object positions. Exploration failed.")
            return

        print(f"[Exploration] Light switch found: {light_switch_name} at {light_switch_pos}")

        # Get robot's current location (assuming positions has 'robot' or similar)
        robot_location = None
        for obj_name in positions:
            if "robot" in obj_name:
                robot_location = positions[obj_name]
                robot_name = obj_name
                break

        if robot_location is None:
            print("[Exploration] Robot location not found. Exploration failed.")
            return

        print(f"[Exploration] Robot initial location: {robot_name} at {robot_location}")

        # === Move robot to light switch location if not already there ===
        # Use execute_go skill if available
        try:
            if hasattr(env, 'current_location'):
                current_loc = env.current_location
            else:
                current_loc = robot_location

            # If robot is not at the light switch, move there
            if not np.allclose(np.array(current_loc), np.array(light_switch_pos), atol=0.05):
                print(f"[Exploration] Moving robot from {current_loc} to light switch at {light_switch_pos}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=current_loc,
                    to_location=light_switch_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after moving to light switch!")
                    return
                env.current_location = light_switch_pos
            else:
                print("[Exploration] Robot already at light switch location.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")
            return

        # === Exploration complete: robot-at light_switch predicate is now true ===
        print("[Exploration] Predicate (robot-at light_switch) established.")

        # === Main Task Plan Execution ===
        # Example: Turn on the light using execute_push_switch
        try:
            print("[Task] Executing execute_push_switch to turn on the light.")
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name=light_switch_name,
                location=light_switch_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after turning on the light!")
                return
        except Exception as e:
            print(f"[Task] Exception during execute_push_switch: {e}")
            return

        # === Continue with further plan steps as needed ===
        # For example, pick up an object, open a drawer, etc.
        # Use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # Example (commented out):
        # obs, reward, done = execute_pick(env, task, object_name, location, ...)
        # if done:
        #     print("[Task] Task ended after picking object!")
        #     return

        print("[Task] Skeleton task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
