# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at robot_location)
        # The missing predicate is likely related to the robot's location.
        # We will attempt to use the available skills to explore and confirm the robot's location predicate.

        # Since we do not have the actual object and location names, we will use placeholders.
        # Replace these with actual names from your environment if available.
        try:
            # Attempt to get robot's current location from positions or observation
            robot_location = None
            if 'robot' in positions:
                robot_location = positions['robot']
            elif 'robot_location' in positions:
                robot_location = positions['robot_location']
            else:
                # Fallback: try to extract from obs or descriptions if possible
                robot_location = None  # Could be set from obs if structure is known

            # For demonstration, we assume there are at least two locations to explore
            # and that positions contains location keys.
            location_keys = [k for k in positions.keys() if 'location' in k or 'room' in k]
            if len(location_keys) < 2:
                print("[Exploration] Not enough locations found for exploration.")
            else:
                from_location = location_keys[0]
                to_location = location_keys[1]

                print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} to test location predicate.")

                # Use the available skill 'execute_go' to move the robot
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location,
                        to_location
                    )
                    print(f"[Exploration] Robot moved from {from_location} to {to_location}.")
                except Exception as e:
                    print(f"[Exploration] Error during execute_go: {e}")

                # After moving, check if the robot's location predicate is updated
                # (In a real system, you would check the state or observation here)
                # For now, just print a message
                print("[Exploration] Check if (robot-at ...) predicate is now true for the new location.")

        except Exception as e:
            print(f"[Exploration] Exception during exploration phase: {e}")

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan step-by-step using only the available skills.
        # For demonstration, we show a generic plan using the available skills.

        # Example: Pick up an object and place it in a drawer (if such objects exist)
        try:
            # Find an object on the floor and a drawer in the environment
            object_on_floor = None
            drawer = None
            for obj_name in positions:
                if 'object' in obj_name or 'ball' in obj_name:
                    object_on_floor = obj_name
                    break
            for obj_name in positions:
                if 'drawer' in obj_name:
                    drawer = obj_name
                    break

            if object_on_floor and drawer:
                # Assume we have a location for the object and the drawer
                object_location = positions.get(object_on_floor)
                drawer_location = positions.get(drawer)

                # Move to the object's location
                if object_location and drawer_location:
                    print(f"[Task] Moving to object {object_on_floor} at {object_location}")
                    try:
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=to_location,  # Use the last location as current
                            to_location=object_location
                        )
                    except Exception as e:
                        print(f"[Task] Error during execute_go to object: {e}")

                    # Pick up the object
                    print(f"[Task] Picking up {object_on_floor}")
                    try:
                        obs, reward, done = execute_pick(
                            env,
                            task,
                            object_on_floor,
                            object_location
                        )
                    except Exception as e:
                        print(f"[Task] Error during execute_pick: {e}")

                    # Move to the drawer's location
                    print(f"[Task] Moving to drawer {drawer} at {drawer_location}")
                    try:
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=object_location,
                            to_location=drawer_location
                        )
                    except Exception as e:
                        print(f"[Task] Error during execute_go to drawer: {e}")

                    # Place the object in the drawer
                    print(f"[Task] Placing {object_on_floor} in {drawer}")
                    try:
                        obs, reward, done = execute_place(
                            env,
                            task,
                            object_on_floor,
                            drawer,
                            drawer_location
                        )
                    except Exception as e:
                        print(f"[Task] Error during execute_place: {e}")

                    if done:
                        print("[Task] Task completed after placing object in drawer!")
                        return
                else:
                    print("[Task] Could not determine object or drawer locations.")
            else:
                print("[Task] No suitable object or drawer found in positions.")

        except Exception as e:
            print(f"[Task] Exception during main plan execution: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
