# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at robot_location)
        # The missing predicate is likely related to the robot's location.
        # In the main domain, the predicate is (robot-at ?p - location)
        # In the exploration domain, the predicate is (robot-at ?r - robot ?loc - location)
        # We need to explore and confirm the robot's location predicate.

        # For exploration, we will attempt to use the available skills to move the robot and observe the effect.
        # Since we cannot define new skills, we use the available ones: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # Let's try to move the robot using execute_go and check if the robot's location changes as expected.
        # We'll print out the robot's location before and after the move to help identify the predicate.

        # Get all locations from positions (assuming keys like 'robot', 'location1', etc.)
        # For demonstration, we assume there are at least two locations: 'location1' and 'location2'
        # and the robot starts at 'robot_location' or similar.

        # Find robot and locations
        robot_location = None
        location_names = []
        for name, pos in positions.items():
            if 'robot' in name:
                robot_location = name
            elif 'location' in name or 'room' in name:
                location_names.append(name)
        if robot_location is None and 'robot_location' in positions:
            robot_location = 'robot_location'
        if not location_names:
            # Fallback: use all keys except robot
            location_names = [k for k in positions.keys() if k != robot_location]

        # Print initial robot location
        print("[Exploration] Initial robot location:", robot_location)
        print("[Exploration] Known locations:", location_names)

        # Try to move the robot to another location using execute_go
        if len(location_names) >= 2:
            from_location = location_names[0]
            to_location = location_names[1]
        elif len(location_names) == 1:
            from_location = location_names[0]
            to_location = location_names[0]
        else:
            print("[Exploration] No locations found for exploration.")
            from_location = None
            to_location = None

        # If possible, perform exploration move
        if from_location and to_location:
            try:
                print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location,
                    to_location
                )
                print(f"[Exploration] Robot moved from {from_location} to {to_location}.")
            except Exception as e:
                print(f"[Exploration] Error during execute_go: {e}")

            # After move, check robot's location again
            positions_after = get_object_positions()
            robot_location_after = None
            for name, pos in positions_after.items():
                if 'robot' in name:
                    robot_location_after = name
            if robot_location_after is None and 'robot_location' in positions_after:
                robot_location_after = 'robot_location'
            print("[Exploration] Robot location after move:", robot_location_after)
        else:
            print("[Exploration] Not enough locations to perform execute_go.")

        # === End of Exploration Phase ===

        # === Main Task Plan (Placeholder) ===
        # Here you would implement the oracle plan using the available skills.
        # For example:
        # obs, reward, done = execute_pick(env, task, object_name, location_name)
        # obs, reward, done = execute_place(env, task, object_name, drawer_name, location_name)
        # etc.
        # Since the actual plan is not provided, this section is left as a placeholder.

        print("[Task] Exploration phase complete. Use the above information to update your domain or plan as needed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()