# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available skills to explore the environment and try to discover the missing predicate.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects/drawers and observe the effects.

        # Get all objects, drawers, and locations from positions
        # (Assume positions dict contains keys: 'objects', 'drawers', 'handles', 'locations')
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        handles = positions.get('handles', [])
        locations = positions.get('locations', [])
        robot_location = positions.get('robot', None)

        # If not provided, try to infer from positions
        if not objects:
            objects = [k for k in positions.keys() if k.startswith('object')]
        if not drawers:
            drawers = [k for k in positions.keys() if k.startswith('drawer')]
        if not handles:
            handles = [k for k in positions.keys() if k.startswith('handle')]
        if not locations:
            locations = [k for k in positions.keys() if k.startswith('location') or k.startswith('room')]
        if not robot_location:
            robot_location = positions.get('robot', None)
            if robot_location is None and locations:
                robot_location = locations[0]

        # 1. Try to move to each location (execute_go)
        for loc in locations:
            if loc == robot_location:
                continue
            try:
                print(f"[Exploration] Moving robot from {robot_location} to {loc} using execute_go.")
                obs, reward, done = execute_go(env, task, robot_location, loc)
                robot_location = loc
                if done:
                    print("[Exploration] Task ended during exploration (move)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during execute_go: {e}")

        # 2. Try to pick up each object (execute_pick)
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to pick up {obj} at {robot_location} using execute_pick.")
                obs, reward, done = execute_pick(env, task, obj, robot_location)
                if done:
                    print("[Exploration] Task ended during exploration (pick)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick: {e}")

        # 3. Try to open each drawer (execute_pull)
        for d in drawers:
            # Find handle for this drawer
            handle = None
            for h in handles:
                if positions.get('handle_of', {}).get(h) == d:
                    handle = h
                    break
            if not handle and handles:
                handle = handles[0]
            try:
                # Pick the handle first if not already holding
                print(f"[Exploration] Attempting to pick handle {handle} for drawer {d} at {robot_location}.")
                obs, reward, done = execute_pick(env, task, handle, robot_location)
                if done:
                    print("[Exploration] Task ended during exploration (pick handle)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick (handle): {e}")
            try:
                print(f"[Exploration] Attempting to pull drawer {d} with handle {handle} at {robot_location} using execute_pull.")
                obs, reward, done = execute_pull(env, task, d, handle, robot_location)
                if done:
                    print("[Exploration] Task ended during exploration (pull)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during execute_pull: {e}")

        # 4. Try to place each object in each drawer (execute_place)
        for obj in objects:
            for d in drawers:
                try:
                    print(f"[Exploration] Attempting to place {obj} in drawer {d} at {robot_location} using execute_place.")
                    obs, reward, done = execute_place(env, task, obj, d, robot_location)
                    if done:
                        print("[Exploration] Task ended during exploration (place)!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during execute_place: {e}")

        # 5. Try to push each drawer closed (execute_push)
        for d in drawers:
            try:
                print(f"[Exploration] Attempting to push drawer {d} closed at {robot_location} using execute_push.")
                obs, reward, done = execute_push(env, task, d, robot_location)
                if done:
                    print("[Exploration] Task ended during exploration (push)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during execute_push: {e}")

        # 6. Try to sweep each object (execute_sweep)
        for obj in objects:
            try:
                print(f"[Exploration] Attempting to sweep {obj} at {robot_location} using execute_sweep.")
                obs, reward, done = execute_sweep(env, task, obj, robot_location)
                if done:
                    print("[Exploration] Task ended during exploration (sweep)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during execute_sweep: {e}")

        # 7. Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Attempting to use gripper at {robot_location} using execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
            if done:
                print("[Exploration] Task ended during exploration (gripper)!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        print("[Exploration] Exploration phase completed. If the planner still fails, check logs for missing predicates or unexpected failures.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()