# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List and Existence Check ===
        # Use robust object management and naming conventions
        object_names = [
            "drawer_top_handle",
            "drawer_middle_handle",
            "drawer_bottom_handle",
            "drawer_top_place_left",
            "drawer_top_place_right",
            "drawer_middle_place",
            "drawer_bottom_place_left",
            "drawer_bottom_place_right",
            "dice_1",
            "dice_2",
            "trash",
            "trash_bin"
        ]

        def add_objects(object_list, new_objects):
            for obj in new_objects:
                if obj in object_names:
                    object_list.append(obj)
                else:
                    print(f"Error: Object '{obj}' not found in object list.")
                    return False
            return True

        # Example: Add dice_1 and dice_2, with existence check
        object_list = []
        new_objects = ["dice_1", "dice_2"]
        if not add_objects(object_list, new_objects):
            print("Error adding objects. Exiting task.")
            return

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to explore and determine which predicate is missing (e.g., lock-known, identified, etc.)
        # We'll use the available skills to perform exploration actions.
        # For demonstration, let's try to identify all objects at each location.

        # Find all unique locations from positions
        all_locations = set()
        for obj, pos in positions.items():
            if isinstance(pos, dict) and 'location' in pos:
                all_locations.add(pos['location'])
            elif isinstance(pos, str):
                all_locations.add(pos)
        all_locations = list(all_locations)

        # Assume robot starts at the first location
        if len(all_locations) == 0:
            print("No locations found in object positions. Exiting.")
            return

        robot_location = all_locations[0]

        # Safety: Check for obstacles or missing objects
        for obj in object_list:
            if obj not in positions:
                print(f"Warning: Object '{obj}' not found in positions. Skipping.")
        
        # Exploration: Try to identify objects at each location
        for loc in all_locations:
            if loc != robot_location:
                try:
                    # Use execute_go to move robot
                    print(f"[Exploration] Moving robot from {robot_location} to {loc}")
                    obs, reward, done = execute_go(env, task, robot_location, loc)
                    robot_location = loc
                except Exception as e:
                    print(f"Error during execute_go: {e}")
                    continue
            # Try to identify objects at this location (simulate exploration)
            for obj in object_list:
                obj_pos = positions.get(obj, None)
                if obj_pos is None:
                    continue
                obj_loc = obj_pos['location'] if isinstance(obj_pos, dict) and 'location' in obj_pos else obj_pos
                if obj_loc == loc:
                    try:
                        print(f"[Exploration] Attempting to identify object '{obj}' at location '{loc}'")
                        # Use execute_sweep as a generic exploration skill
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                    except Exception as e:
                        print(f"Error during execute_sweep for {obj}: {e}")

        # === Main Task Plan (Example) ===
        # After exploration, proceed with the main plan using only available skills.
        # For demonstration, pick up dice_1 and place it in drawer_middle_place if possible.

        # Check if dice_1 and drawer_middle_place exist and have positions
        dice_1 = "dice_1"
        drawer_middle_place = "drawer_middle_place"
        if dice_1 in positions and drawer_middle_place in positions:
            dice_1_loc = positions[dice_1]['location'] if isinstance(positions[dice_1], dict) and 'location' in positions[dice_1] else positions[dice_1]
            drawer_middle_place_loc = positions[drawer_middle_place]['location'] if isinstance(positions[drawer_middle_place], dict) and 'location' in positions[drawer_middle_place] else positions[drawer_middle_place]
            # Move to dice_1 location if not already there
            if robot_location != dice_1_loc:
                try:
                    print(f"[Task] Moving to {dice_1_loc} to pick up {dice_1}")
                    obs, reward, done = execute_go(env, task, robot_location, dice_1_loc)
                    robot_location = dice_1_loc
                except Exception as e:
                    print(f"Error during execute_go to {dice_1_loc}: {e}")
            # Pick up dice_1
            try:
                print(f"[Task] Picking up {dice_1} at {dice_1_loc}")
                obs, reward, done = execute_pick(env, task, dice_1, dice_1_loc)
            except Exception as e:
                print(f"Error during execute_pick for {dice_1}: {e}")
            # Move to drawer_middle_place location
            if robot_location != drawer_middle_place_loc:
                try:
                    print(f"[Task] Moving to {drawer_middle_place_loc} to place {dice_1}")
                    obs, reward, done = execute_go(env, task, robot_location, drawer_middle_place_loc)
                    robot_location = drawer_middle_place_loc
                except Exception as e:
                    print(f"Error during execute_go to {drawer_middle_place_loc}: {e}")
            # Place dice_1 in drawer_middle_place (assuming drawer is open and not full)
            try:
                print(f"[Task] Placing {dice_1} in {drawer_middle_place} at {drawer_middle_place_loc}")
                obs, reward, done = execute_place(env, task, dice_1, drawer_middle_place, drawer_middle_place_loc)
            except Exception as e:
                print(f"Error during execute_place for {dice_1}: {e}")
        else:
            print("dice_1 or drawer_middle_place not found in positions. Skipping main task.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
