# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use available exploration skills to probe the environment and infer missing predicates.
        # The exploration domain provides skills like execute_go_identify, execute_go_temperature, execute_pick_weight, execute_pick_durability, execute_pull.
        # However, in this environment, only the following skills are available:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        # We will use these to explore the environment and check for missing predicates.

        # For demonstration, we will attempt to use each available skill in a safe, exploratory way.
        # This will help us determine if any action fails due to a missing predicate, which we can then report.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Example: Try to pick up each object on the floor at its location
        # and try to open/close drawers, move, etc., to see if any predicate is missing.

        # Get all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('drawer') and not name.startswith('location')]
        drawer_names = [name for name in positions if name.startswith('drawer')]
        location_names = [name for name in positions if name.startswith('location')]

        # Try to pick each object (simulate exploration)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                # Find the nearest location for the object (if available)
                # For simplicity, assume the object's position is its location
                # In a real scenario, you would map positions to location names
                location = None
                for loc in location_names:
                    if np.allclose(positions[loc], obj_pos, atol=0.1):
                        location = loc
                        break
                if location is None and location_names:
                    location = location_names[0]  # fallback

                print(f"[Exploration] Trying to pick {obj} at {location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    location
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj}.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj}: {e}")

        # Try to open each drawer (simulate exploration)
        for drawer in drawer_names:
            try:
                # Find handle object for the drawer (if any)
                handle_obj = None
                for obj in object_names:
                    if 'handle' in obj and drawer in obj:
                        handle_obj = obj
                        break
                if handle_obj is None and object_names:
                    handle_obj = object_names[0]  # fallback

                # Find a location for the drawer
                drawer_pos = positions[drawer]
                location = None
                for loc in location_names:
                    if np.allclose(positions[loc], drawer_pos, atol=0.1):
                        location = loc
                        break
                if location is None and location_names:
                    location = location_names[0]

                print(f"[Exploration] Trying to pull (open) {drawer} with handle {handle_obj} at {location}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer,
                    handle_obj,
                    location
                )
                if done:
                    print(f"[Exploration] Successfully opened {drawer}.")
            except Exception as e:
                print(f"[Exploration] Failed to open {drawer}: {e}")

        # Try to close each drawer (simulate exploration)
        for drawer in drawer_names:
            try:
                drawer_pos = positions[drawer]
                location = None
                for loc in location_names:
                    if np.allclose(positions[loc], drawer_pos, atol=0.1):
                        location = loc
                        break
                if location is None and location_names:
                    location = location_names[0]

                print(f"[Exploration] Trying to push (close) {drawer} at {location}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer,
                    location
                )
                if done:
                    print(f"[Exploration] Successfully closed {drawer}.")
            except Exception as e:
                print(f"[Exploration] Failed to close {drawer}: {e}")

        # Try to move between locations (simulate exploration)
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Trying to move from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_loc,
                    to_loc
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")

        # Try to sweep each object (simulate exploration)
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                location = None
                for loc in location_names:
                    if np.allclose(positions[loc], obj_pos, atol=0.1):
                        location = loc
                        break
                if location is None and location_names:
                    location = location_names[0]

                print(f"[Exploration] Trying to sweep {obj} at {location}")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    location
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
            except Exception as e:
                print(f"[Exploration] Failed to sweep {obj}: {e}")

        # Try to use gripper (simulate exploration)
        try:
            print(f"[Exploration] Trying to use gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")

        # Try to rotate (simulate exploration)
        try:
            print(f"[Exploration] Trying to rotate")
            obs, reward, done = execute_rotate(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully rotated.")
        except Exception as e:
            print(f"[Exploration] Failed to rotate: {e}")

        print("[Exploration] Exploration phase complete. If any action failed due to a missing predicate, check the above logs for clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # If you have an oracle plan, you would execute it here step by step using the available skills.
        # For this task, the focus is on exploration to identify missing predicates.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()