# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (drawer-unlocked drawer1) ===
        # The feedback indicates that the predicate (drawer-unlocked drawer1) is missing.
        # We need to explore the environment to determine the lock state of the drawer.

        # For this, we will use the available skills to:
        # 1. Move the robot to the drawer's location.
        # 2. Attempt to pull the drawer handle (execute_pull) to check if it is unlocked.
        # 3. If the drawer can be pulled, we infer it is unlocked.

        # --- Example object and location names (replace with actual names from your environment) ---
        # Let's assume:
        #   - drawer1 is the drawer of interest
        #   - handle1 is the handle object attached to drawer1
        #   - location_drawer is the location of the drawer
        #   - robot starts at location_start

        # You may need to adjust these names based on your environment's object naming.
        try:
            # Example object keys (replace with actual keys from positions if needed)
            drawer_name = 'drawer1'
            handle_name = 'handle1'
            location_drawer = 'drawer_area'
            location_start = 'start_area'

            # If your positions dict uses different keys, update accordingly
            if drawer_name not in positions or handle_name not in positions:
                # Try to infer from available keys
                for k in positions.keys():
                    if 'drawer' in k:
                        drawer_name = k
                    if 'handle' in k:
                        handle_name = k
                # Fallback: use the first available location as start
                location_start = list(positions.keys())[0]
                location_drawer = drawer_name

            print(f"[Exploration] Moving robot from {location_start} to {location_drawer} to check lock state.")

            # 1. Move robot to drawer location (execute_go)
            obs, reward, done = execute_go(
                env,
                task,
                from_location=location_start,
                to_location=location_drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )

            if done:
                print("[Exploration] Robot failed to reach drawer location.")
                return

            # 2. Pick the handle (execute_pick)
            print(f"[Exploration] Attempting to pick handle {handle_name} at {location_drawer}.")
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,
                location=location_drawer,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )

            if done:
                print("[Exploration] Failed to pick handle.")
                return

            # 3. Attempt to pull the drawer (execute_pull)
            print(f"[Exploration] Attempting to pull drawer {drawer_name} using handle {handle_name}.")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer=drawer_name,
                handle=handle_name,
                location=location_drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )

            if done:
                print(f"[Exploration] Drawer {drawer_name} could not be pulled. It may be locked.")
                print("[Exploration] Predicate (drawer-unlocked {drawer_name}) is likely FALSE.")
            else:
                print(f"[Exploration] Drawer {drawer_name} was pulled open. It is unlocked.")
                print(f"[Exploration] Predicate (drawer-unlocked {drawer_name}) is TRUE.")

            # Optionally, push the drawer closed again (execute_push)
            print(f"[Exploration] Attempting to push drawer {drawer_name} closed.")
            obs, reward, done = execute_push(
                env,
                task,
                drawer=drawer_name,
                location=location_drawer,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )

        except Exception as e:
            print(f"[Exploration] Exception during exploration: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after exploration, using the discovered predicate.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
