# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object and Location Names (from observation and PDDL) ---
        dice_names = ['dice1', 'dice2']
        drawer_name = 'drawer1'
        handle_name = 'handle1'
        floor_loc = 'floor'
        drawer_area_loc = 'drawer-area'
        ready_pose_loc = 'ready-pose'

        # --- Exploration Phase: Identify Missing Predicate ---
        # The feedback suggests a missing predicate is involved in storing dice in the open drawer.
        # We'll perform an exploration step to check the state of the drawer and objects.

        # 1. Move robot to ready-pose if not already there
        try:
            robot_pos = positions.get('robot', ready_pose_loc)
            if robot_pos != ready_pose_loc:
                print("[Exploration] Moving robot to ready-pose.")
                obs, reward, done = execute_go(env, task, robot_pos, ready_pose_loc)
        except Exception as e:
            print(f"[Exploration] Could not determine robot position: {e}")

        # 2. Check if drawer1 is open; if not, open it
        try:
            # We assume positions or task.get_state() can tell us if drawer1 is open
            drawer_open = False
            if hasattr(task, 'is_drawer_open'):
                drawer_open = task.is_drawer_open(drawer_name)
            elif 'drawer1_open' in positions:
                drawer_open = positions['drawer1_open']
            else:
                # Fallback: try to open anyway
                drawer_open = False

            if not drawer_open:
                print("[Exploration] Drawer1 is not open. Attempting to open it.")
                # Pick up the handle
                obs, reward, done = execute_pick(env, task, handle_name, floor_loc)
                # Pull the handle to open the drawer
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, ready_pose_loc)
            else:
                print("[Exploration] Drawer1 is already open.")
        except Exception as e:
            print(f"[Exploration] Error checking/opening drawer: {e}")

        # 3. Exploration: Check if dice are on the floor and can be picked
        for dice in dice_names:
            try:
                dice_on_floor = False
                if hasattr(task, 'is_on_floor'):
                    dice_on_floor = task.is_on_floor(dice)
                elif f'{dice}_on_floor' in positions:
                    dice_on_floor = positions[f'{dice}_on_floor']
                else:
                    # Fallback: assume on floor as per initial state
                    dice_on_floor = True

                if dice_on_floor:
                    print(f"[Exploration] {dice} is on the floor. Ready to pick.")
                else:
                    print(f"[Exploration] {dice} is not on the floor.")
            except Exception as e:
                print(f"[Exploration] Error checking {dice} position: {e}")

        # --- Main Task Plan: Store Both Dice in the Open Drawer ---

        for dice in dice_names:
            try:
                # 1. Pick up the dice from the floor
                print(f"[Task] Picking up {dice} from the floor.")
                obs, reward, done = execute_pick(env, task, dice, floor_loc)
            except Exception as e:
                print(f"[Task] Error picking up {dice}: {e}")
                continue

            try:
                # 2. Place the dice into the open drawer1 at ready-pose
                print(f"[Task] Placing {dice} into {drawer_name}.")
                obs, reward, done = execute_place(env, task, dice, drawer_name, ready_pose_loc)
            except Exception as e:
                print(f"[Task] Error placing {dice} into {drawer_name}: {e}")
                continue

        # --- Optional: Close the Drawer (if required by the goal) ---
        try:
            # Check if drawer should be closed (not required by initial goal, but possible)
            if hasattr(task, 'should_close_drawer') and task.should_close_drawer(drawer_name):
                print(f"[Task] Closing {drawer_name}.")
                obs, reward, done = execute_push(env, task, drawer_name, ready_pose_loc)
        except Exception as e:
            print(f"[Task] Error closing {drawer_name}: {e}")

        print("[Task] All dice stored in the drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
