# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (robot-free) ===
        # The feedback indicates that the predicate (robot-free) is missing or not being set as expected.
        # We will use the available skills to explore the environment and check the robot's state after each action.

        # For demonstration, we will attempt to use the available skills in a sequence that would
        # allow us to observe the effect on the robot's state, especially regarding (robot-free).

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to:
        # 1. Move the robot to a location (execute_go)
        # 2. Pick up an object (execute_pick)
        # 3. Place the object (execute_place)
        # 4. Push or pull a drawer (execute_push, execute_pull)
        # 5. Check if the robot is free after each action

        # For this, we need to select some objects, locations, and drawers from the positions dictionary.
        # We'll use try/except to handle missing keys gracefully.

        try:
            # Example: Find an object on the floor, a drawer, and two locations
            object_name = None
            drawer_name = None
            handle_name = None
            location_names = []
            for name, info in positions.items():
                if 'type' in info:
                    if info['type'] == 'object' and object_name is None:
                        object_name = name
                    if info['type'] == 'drawer' and drawer_name is None:
                        drawer_name = name
                    if info['type'] == 'handle' and handle_name is None:
                        handle_name = name
                if 'type' in info and info['type'] == 'location':
                    location_names.append(name)
            # Fallbacks if not found
            if object_name is None:
                object_name = list(positions.keys())[0]
            if drawer_name is None:
                drawer_name = list(positions.keys())[1]
            if handle_name is None:
                handle_name = list(positions.keys())[2]
            if len(location_names) < 2:
                location_names = list(positions.keys())[:2]
            from_location = location_names[0]
            to_location = location_names[1]
        except Exception as e:
            print("[Exploration] Error finding objects/locations:", e)
            return

        # 1. Move the robot to the starting location (if not already there)
        try:
            print(f"[Exploration] Moving robot from {from_location} to {to_location} using execute_go")
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            if done:
                print("[Exploration] Task ended after execute_go!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_go:", e)

        # 2. Try to pick up the object
        try:
            print(f"[Exploration] Picking up {object_name} at {to_location} using execute_pick")
            obs, reward, done = execute_pick(
                env,
                task,
                object_name,
                to_location
            )
            if done:
                print("[Exploration] Task ended after execute_pick!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_pick:", e)

        # 3. Try to place the object in the drawer
        try:
            print(f"[Exploration] Placing {object_name} in {drawer_name} at {to_location} using execute_place")
            obs, reward, done = execute_place(
                env,
                task,
                object_name,
                drawer_name,
                to_location
            )
            if done:
                print("[Exploration] Task ended after execute_place!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_place:", e)

        # 4. Try to pull the drawer open using the handle
        try:
            print(f"[Exploration] Pulling {drawer_name} with handle {handle_name} at {to_location} using execute_pull")
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name,
                handle_name,
                to_location
            )
            if done:
                print("[Exploration] Task ended after execute_pull!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_pull:", e)

        # 5. Try to push the drawer closed
        try:
            print(f"[Exploration] Pushing {drawer_name} at {to_location} using execute_push")
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name,
                to_location
            )
            if done:
                print("[Exploration] Task ended after execute_push!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_push:", e)

        # 6. Try to sweep the object (if available)
        try:
            print(f"[Exploration] Sweeping {object_name} at {to_location} using execute_sweep")
            obs, reward, done = execute_sweep(
                env,
                task,
                object_name,
                to_location
            )
            if done:
                print("[Exploration] Task ended after execute_sweep!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_sweep:", e)

        # 7. Try to use the gripper (if available)
        try:
            print(f"[Exploration] Using gripper with execute_gripper")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print("[Exploration] Task ended after execute_gripper!")
                return
        except Exception as e:
            print("[Exploration] Error during execute_gripper:", e)

        # After each action, you may want to check the robot's state (e.g., is it free? is it holding something?)
        # This can be done by inspecting the observation or task state if such an API is available.

        print("[Exploration] Exploration phase complete. Check logs and observations for (robot-free) predicate status.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()