# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment provides feedback about missing knowledge.

        # Get all objects, drawers, and locations from positions (assuming keys are descriptive)
        object_names = [name for name in positions if 'object' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        handle_names = [name for name in positions if 'handle' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'table' in name]

        # Fallback: If no explicit location names, use all unique position values as locations
        if not location_names:
            location_names = list(set([v for v in positions.values()]))

        # Try to move to each location and interact with objects/drawers
        # This will help us discover if any knowledge predicate is missing (e.g., lock-known, identified, etc.)
        # Use only available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # 1. Try to move to each location
        for loc in location_names:
            try:
                # Find current robot location (if available)
                # For this skeleton, we assume the robot starts at the first location
                current_loc = location_names[0]
                if loc != current_loc:
                    print(f"[Exploration] Moving robot from {current_loc} to {loc}")
                    obs, reward, done = execute_go(env, task, current_loc, loc)
                    current_loc = loc
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")

            # 2. Try to pick up each object at this location
            for obj in object_names:
                try:
                    print(f"[Exploration] Attempting to pick {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")

            # 3. Try to interact with each drawer at this location
            for drawer in drawer_names:
                # Try to pull the drawer using its handle (if handle exists)
                handle = None
                for h in handle_names:
                    # Assume handle-of relation is encoded in the name, e.g., handle1_of_drawer1
                    if drawer in h:
                        handle = h
                        break
                if handle:
                    try:
                        # First, pick the handle
                        print(f"[Exploration] Attempting to pick handle {handle} at {loc}")
                        obs, reward, done = execute_pick(env, task, handle, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to pick handle {handle} at {loc}: {e}")
                    try:
                        print(f"[Exploration] Attempting to pull drawer {drawer} with handle {handle} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to pull drawer {drawer} with handle {handle} at {loc}: {e}")
                else:
                    # If no handle, try to pull the drawer directly (if allowed)
                    try:
                        print(f"[Exploration] Attempting to pull drawer {drawer} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, None, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to pull drawer {drawer} at {loc}: {e}")

            # 4. Try to sweep any object at this location
            for obj in object_names:
                try:
                    print(f"[Exploration] Attempting to sweep {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")

        # 5. Try to push each drawer (close it)
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to push (close) drawer {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to push drawer {drawer} at {loc}: {e}")

        # 6. Try to place each object into each drawer at each location
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Attempting to place {obj} into {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} into {drawer} at {loc}: {e}")

        # 7. Try to use gripper skill (if any)
        try:
            print(f"[Exploration] Attempting to use gripper skill")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper skill: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()