# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.

        # We will attempt to use the available skills to interact with objects and drawers,
        # and observe if any action fails or if the environment state is not as expected,
        # which may indicate a missing predicate (e.g., lock-known).

        # Get all objects, drawers, and locations from the positions dictionary
        # (Assume positions dict keys are like: 'object_1', 'drawer_1', 'location_1', etc.)
        object_names = [name for name in positions if name.startswith('object')]
        drawer_names = [name for name in positions if name.startswith('drawer')]
        location_names = [name for name in positions if name.startswith('location')]

        # For demonstration, we will try to:
        # 1. Move to each location
        # 2. Try to pick each object
        # 3. Try to open each drawer (pull)
        # 4. Try to place an object in a drawer
        # 5. Try to push (close) each drawer

        # Keep track of which predicates might be missing based on action failures
        missing_predicates = set()

        # Helper: get current robot location (assume task or env provides this info)
        def get_robot_location():
            # This is a placeholder; in practice, get from env/task/obs
            # For now, just return the first location
            if location_names:
                return location_names[0]
            return None

        # 1. Move to each location using execute_go
        for to_loc in location_names:
            from_loc = get_robot_location()
            if from_loc == to_loc:
                continue
            try:
                print(f"[Exploration] Moving from {from_loc} to {to_loc} using execute_go")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
            except Exception as e:
                print(f"[Exploration] Failed to move from {from_loc} to {to_loc}: {e}")
                missing_predicates.add('robot-at')
            # Update robot location (simulate)
            # In real code, update from obs or env
            # For this exploration, just continue

        # 2. Try to pick each object using execute_pick
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc} using execute_pick")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to pick {obj} at {loc}: {e}")
                    # Check for missing predicates based on error message or context
                    if 'on-floor' in str(e):
                        missing_predicates.add('on-floor')
                    if 'hand-empty' in str(e):
                        missing_predicates.add('hand-empty')
                    if 'robot-free' in str(e):
                        missing_predicates.add('robot-free')
                    if 'robot-at' in str(e):
                        missing_predicates.add('robot-at')
                    # Continue to next object/location

        # 3. Try to open each drawer using execute_pull
        for drawer in drawer_names:
            # Find handle object for this drawer (if any)
            handle_obj = None
            for obj in object_names:
                # Assume positions or env can tell us if obj is a handle for drawer
                # For now, just pick the first object as handle
                handle_obj = obj
                break
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pull {drawer} with handle {handle_obj} at {loc} using execute_pull")
                    obs, reward, done = execute_pull(env, task, drawer, handle_obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to pull {drawer} with handle {handle_obj} at {loc}: {e}")
                    # Check for missing predicates
                    if 'drawer-unlocked' in str(e):
                        missing_predicates.add('drawer-unlocked')
                    if 'drawer-closed' in str(e):
                        missing_predicates.add('drawer-closed')
                    if 'handle-of' in str(e):
                        missing_predicates.add('handle-of')
                    if 'holding' in str(e):
                        missing_predicates.add('holding')
                    if 'robot-at' in str(e):
                        missing_predicates.add('robot-at')
                    # If error suggests lock-known is missing, add it
                    if 'lock-known' in str(e):
                        missing_predicates.add('lock-known')

        # 4. Try to place each object in each drawer using execute_place
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to place {obj} in {drawer} at {loc} using execute_place")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration] Failed to place {obj} in {drawer} at {loc}: {e}")
                        # Check for missing predicates
                        if 'holding' in str(e):
                            missing_predicates.add('holding')
                        if 'drawer-open' in str(e):
                            missing_predicates.add('drawer-open')
                        if 'drawer-full' in str(e):
                            missing_predicates.add('drawer-full')
                        if 'robot-at' in str(e):
                            missing_predicates.add('robot-at')

        # 5. Try to push (close) each drawer using execute_push
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to push (close) {drawer} at {loc} using execute_push")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to push {drawer} at {loc}: {e}")
                    # Check for missing predicates
                    if 'drawer-open' in str(e):
                        missing_predicates.add('drawer-open')
                    if 'robot-free' in str(e):
                        missing_predicates.add('robot-free')
                    if 'robot-at' in str(e):
                        missing_predicates.add('robot-at')

        # 6. Try to sweep each object using execute_sweep
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc} using execute_sweep")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] Failed to sweep {obj} at {loc}: {e}")
                    # Check for missing predicates
                    if 'on-floor' in str(e):
                        missing_predicates.add('on-floor')
                    if 'robot-free' in str(e):
                        missing_predicates.add('robot-free')
                    if 'robot-at' in str(e):
                        missing_predicates.add('robot-at')

        # 7. Try to use gripper using execute_gripper
        try:
            print(f"[Exploration] Trying to use gripper using execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] Failed to use gripper: {e}")
            if 'robot-free' in str(e):
                missing_predicates.add('robot-free')

        # === Exploration Summary ===
        if missing_predicates:
            print("[Exploration] Potential missing predicates detected:")
            for pred in missing_predicates:
                print(f"  - {pred}")
        else:
            print("[Exploration] No missing predicates detected during skill execution.")

        # === End of Exploration Phase ===
        # At this point, you can use the exploration results to update your domain/problem files,
        # or to inform the next planning phase.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()