# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Define object and location names as per the problem ===
        # These must match the names in the PDDL and the environment
        object_list = ['dice1', 'dice2', 'handle1', 'handle2', 'handle3']
        drawer_list = ['drawer1', 'drawer2', 'drawer3']
        location_list = ['floor', 'drawer-area', 'ready-pose']

        # Map handles to drawers
        handle_to_drawer = {
            'handle1': 'drawer1',
            'handle2': 'drawer2',
            'handle3': 'drawer3'
        }

        # === Exploration Phase: Identify missing predicates ===
        # According to feedback, we need to ensure only objects in object_list are used
        # and that the robot can interact with them. We also need to check for missing predicates.
        # For this, we can attempt to interact with each object and check for errors.

        # Exploration: Check which objects are present and can be interacted with
        print("[Exploration] Checking interactable objects in object_list...")
        interactable_objects = []
        for obj in object_list:
            if obj in positions:
                interactable_objects.append(obj)
            else:
                print(f"[Exploration] Warning: {obj} not found in environment positions.")

        # Exploration: Check for missing predicates by attempting to perform a skill
        # For example, try to pick each object from the floor at ready-pose
        print("[Exploration] Attempting to pick each object to check for missing predicates...")
        for obj in interactable_objects:
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=obj,
                    location='ready-pose'
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj} at ready-pose.")
                else:
                    print(f"[Exploration] Could not pick {obj} at ready-pose. Possible missing predicate or precondition.")
            except Exception as e:
                print(f"[Exploration] Exception when picking {obj}: {e}")

        # === Oracle Plan Execution ===
        # The plan is to put dice1 and dice2 into drawer1, which is open and empty.
        # The robot starts at ready-pose, hand-empty, robot-free.

        # Step 1: Move to ready-pose (already at ready-pose, but ensure position)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location='ready-pose',
                to_location='ready-pose'
            )
            print("[Plan] Robot moved to ready-pose.")
        except Exception as e:
            print(f"[Plan] Exception during move to ready-pose: {e}")

        # Step 2: (Drawer1 is already open and empty per initial state, so skip open)

        # Step 3: Pick dice1 from floor at ready-pose
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj='dice1',
                location='ready-pose'
            )
            print("[Plan] Picked up dice1.")
        except Exception as e:
            print(f"[Plan] Exception during pick dice1: {e}")

        # Step 4: Place dice1 into drawer1 at ready-pose
        try:
            obs, reward, done = execute_place(
                env,
                task,
                obj='dice1',
                drawer='drawer1',
                location='ready-pose'
            )
            print("[Plan] Placed dice1 into drawer1.")
        except Exception as e:
            print(f"[Plan] Exception during place dice1: {e}")

        # Step 5: Pick dice2 from floor at ready-pose
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj='dice2',
                location='ready-pose'
            )
            print("[Plan] Picked up dice2.")
        except Exception as e:
            print(f"[Plan] Exception during pick dice2: {e}")

        # Step 6: Place dice2 into drawer1 at ready-pose
        try:
            obs, reward, done = execute_place(
                env,
                task,
                obj='dice2',
                drawer='drawer1',
                location='ready-pose'
            )
            print("[Plan] Placed dice2 into drawer1.")
        except Exception as e:
            print(f"[Plan] Exception during place dice2: {e}")

        # === End of Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
