# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will use the available exploration actions to probe the environment and
        # determine which predicate is missing (e.g., lock-known, weight-known, etc.).
        #
        # We use only the available skills: ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']
        #
        # The exploration domain suggests that 'execute_pull' can be used to discover 'lock-known' for an object.
        # We'll attempt to pull on all objects that look like handles or drawers to see if the lock state is known.

        # For demonstration, we assume the following conventions for object/drawer/handle names:
        # - All objects, drawers, and handles are present in the 'positions' dictionary.
        # - Each drawer has a handle, and the mapping is available via a 'handle-of' predicate or naming convention.

        # Step 1: Find all drawers and handles
        drawers = []
        handles = []
        objects = []
        for name in positions:
            if 'drawer' in name:
                drawers.append(name)
            elif 'handle' in name:
                handles.append(name)
            else:
                objects.append(name)

        # Step 2: Try to pull on each handle to check for lock-known predicate
        # We assume the robot starts at some initial location; get it from the observation if possible
        robot_location = None
        for key in positions:
            if 'robot' in key or 'base' in key:
                robot_location = key
                break
        if robot_location is None:
            # Fallback: use a default location if not found
            robot_location = 'robot_base'

        # For each drawer, try to pull its handle
        for drawer in drawers:
            # Find the handle for this drawer
            handle = None
            for h in handles:
                if drawer in h or h in drawer:
                    handle = h
                    break
            if handle is None:
                continue  # Skip if no handle found

            # Move to the drawer's location if needed
            try:
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=drawer)
                robot_location = drawer
            except Exception as e:
                print(f"[Exploration] Failed to move to {drawer}: {e}")

            # Try to pick the handle
            try:
                obs, reward, done = execute_pick(env, task, obj=handle, location=drawer)
            except Exception as e:
                print(f"[Exploration] Failed to pick handle {handle} at {drawer}: {e}")

            # Try to pull the drawer (this may reveal lock-known)
            try:
                obs, reward, done = execute_pull(env, task, drawer=drawer, handle=handle, location=drawer)
                print(f"[Exploration] Pulled {drawer} using {handle} at {drawer}.")
            except Exception as e:
                print(f"[Exploration] Failed to pull {drawer} with {handle}: {e}")

            # Place the handle back if needed (optional)
            try:
                obs, reward, done = execute_place(env, task, obj=handle, drawer=drawer, location=drawer)
            except Exception as e:
                pass  # Not all handles need to be placed back

        # Step 3: Try to pick and place other objects to check for other missing predicates
        for obj in objects:
            # Move to object's location if needed
            try:
                obs, reward, done = execute_go(env, task, from_location=robot_location, to_location=obj)
                robot_location = obj
            except Exception as e:
                print(f"[Exploration] Failed to move to {obj}: {e}")

            # Try to pick the object
            try:
                obs, reward, done = execute_pick(env, task, obj=obj, location=obj)
                print(f"[Exploration] Picked {obj} at {obj}.")
            except Exception as e:
                print(f"[Exploration] Failed to pick {obj} at {obj}: {e}")

            # Try to place the object in a drawer if possible
            for drawer in drawers:
                try:
                    obs, reward, done = execute_place(env, task, obj=obj, drawer=drawer, location=drawer)
                    print(f"[Exploration] Placed {obj} in {drawer}.")
                    break
                except Exception as e:
                    continue

        # Step 4: Try other available skills for completeness
        # Sweep, rotate, gripper, push
        for obj in objects:
            try:
                obs, reward, done = execute_sweep(env, task, obj=obj, location=obj)
            except Exception as e:
                pass
            try:
                obs, reward, done = execute_rotate(env, task, obj=obj, location=obj)
            except Exception as e:
                pass

        for drawer in drawers:
            try:
                obs, reward, done = execute_push(env, task, drawer=drawer, location=drawer)
            except Exception as e:
                pass

        try:
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            pass

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, analyze the results/logs to determine which predicate is missing.
        # This can be done by checking for failed actions and their error messages.
        # Once the missing predicate is identified, update the domain/problem files accordingly.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
