# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback suggests we need to check for object existence before placing in drawer.
        # We'll perform an exploration step to identify all objects at the drawer area.

        # Assume 'drawer_area' is a location key in positions, and 'object_list' is a list of all objects
        # For demonstration, we try to identify all objects at the drawer area

        # Get all object names and drawer names from positions
        object_names = []
        drawer_names = []
        location_names = set()
        for name, info in positions.items():
            if 'type' in info:
                if info['type'] == 'object':
                    object_names.append(name)
                elif info['type'] == 'drawer':
                    drawer_names.append(name)
            if 'location' in info:
                location_names.add(info['location'])

        # For robustness, use a set for object existence checks
        object_set = set(object_names)

        # For this example, assume we want to put all objects on the floor into the first drawer
        if not drawer_names:
            print("[Error] No drawers found in the environment!")
            return
        drawer_name = drawer_names[0]

        # Find the location of the drawer
        drawer_location = None
        for name, info in positions.items():
            if name == drawer_name and 'location' in info:
                drawer_location = info['location']
                break
        if drawer_location is None:
            print(f"[Error] Could not determine location for drawer {drawer_name}")
            return

        # Find all objects on the floor (not in any drawer)
        floor_objects = []
        for name in object_names:
            info = positions.get(name, {})
            if info.get('on_floor', False):
                floor_objects.append(name)

        # === Exploration: Identify objects at the drawer area ===
        # Use execute_go (move) and execute_pick to identify and pick up objects
        # Use execute_place to place them in the drawer

        # Move robot to drawer location if not already there
        robot_location = None
        for name, info in positions.items():
            if name == 'robot' and 'location' in info:
                robot_location = info['location']
                break
        if robot_location != drawer_location:
            try:
                obs, reward, done = execute_go(env, task, robot_location, drawer_location)
                robot_location = drawer_location
            except Exception as e:
                print(f"[Error] Failed to move robot to drawer location: {e}")
                return

        # Check drawer state before placing objects
        # For this example, we assume the drawer must be open and empty
        # We'll use the positions info to check drawer state
        drawer_info = positions.get(drawer_name, {})
        is_drawer_open = drawer_info.get('open', False)
        is_drawer_empty = drawer_info.get('empty', False)
        if not is_drawer_open:
            # Need to open the drawer
            # Find the handle object for the drawer
            handle_name = None
            for name, info in positions.items():
                if info.get('type') == 'handle' and info.get('drawer') == drawer_name:
                    handle_name = name
                    break
            if handle_name is None:
                print(f"[Error] No handle found for drawer {drawer_name}")
                return
            # Pick the handle
            try:
                obs, reward, done = execute_pick(env, task, handle_name, drawer_location)
            except Exception as e:
                print(f"[Error] Failed to pick handle {handle_name}: {e}")
                return
            # Pull to open the drawer
            try:
                obs, reward, done = execute_pull(env, task, drawer_name, handle_name, drawer_location)
                is_drawer_open = True
            except Exception as e:
                print(f"[Error] Failed to pull and open drawer {drawer_name}: {e}")
                return

        # Check if drawer is empty
        if not is_drawer_empty:
            print(f"[Warning] Drawer {drawer_name} is not empty. Skipping placing objects.")
            return

        # Place each object in the drawer, with validation
        for obj in floor_objects:
            if obj not in object_set:
                print(f"[Warning] Object {obj} not in declared object list. Skipping.")
                continue
            # Pick the object
            try:
                obs, reward, done = execute_pick(env, task, obj, drawer_location)
            except Exception as e:
                print(f"[Error] Failed to pick object {obj}: {e}")
                continue
            # Place the object in the drawer
            try:
                obs, reward, done = execute_place(env, task, obj, drawer_name, drawer_location)
                print(f"[Task] Placed object {obj} in drawer {drawer_name}.")
            except Exception as e:
                print(f"[Error] Failed to place object {obj} in drawer {drawer_name}: {e}")
                continue

        # Optionally, close the drawer if required
        try:
            obs, reward, done = execute_push(env, task, drawer_name, drawer_location)
            print(f"[Task] Closed drawer {drawer_name}.")
        except Exception as e:
            print(f"[Warning] Could not close drawer {drawer_name}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()