# run_skeleton_task.py (Completed Executable for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use predefined primitives only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # ======= EXPLORATION PHASE =======
        # The feedback suggests that action with predicate (robot-at handle_pose) needs to be explored.
        # The aim is to find out which predicate is missing via the exploration domain;
        # this means we need to move the robot to 'handle_pose' and try to identify objects or properties at that location.

        # For this, we will perform a go/identify action, and possibly go_pick to cover what predicates become true,
        # mapping to the available skills only. We'll mimic what the plan would do in an oracle-driven fashion.

        # Required skill names:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # ---- 1. Identify 'handle_pose' in the positions dict ----
        if 'handle_pose' in positions:
            handle_pose = positions['handle_pose']
        else:
            print("[Exploration] handle_pose not found in position dictionary.")
            handle_pose = None

        # ---- 2. Determine where the robot is now ----
        robot_location = None
        for key in positions:
            if 'robot' in key or key == 'robot':
                robot_location = positions[key]
                break
        if robot_location is None and 'robot_at' in positions:
            robot_location = positions['robot_at']

        # For this code, assume 'robot_location' is known or default to one of the known positions.
        if robot_location is None:
            print("[Exploration] Robot's initial position unknown, assuming first location in dict.")
            for k, v in positions.items():
                if isinstance(v, tuple) and len(v) == 3:
                    robot_location = v
                    break

        # ---- 3. Exploration: Move to handle_pose ----
        # Use the 'execute_go' skill to move the robot to handle_pose.
        if handle_pose is not None and robot_location is not None:
            try:
                print(f"[Exploration] Moving robot from {robot_location} to handle_pose {handle_pose} ...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_pos=robot_location,
                    to_pos=handle_pose,
                    approach_distance=0.12,
                    max_steps=100,
                    threshold=0.018,
                    approach_axis='z',
                    timeout=10.0
                )
                print("[Exploration] Reached handle_pose.")
                # Update robot's current location
                robot_location = handle_pose
            except Exception as e:
                print(f"[Exploration] Exception in execute_go: {e}")

        # ---- 4. Exploration: Sense what's at handle_pose ----
        # There may be an object at handle_pose (such as a handle object)
        # We'll look to see if any object is at that location and attempt to identify it.
        handle_object = None
        for obj_name, pos in positions.items():
            # Simple proximity check in 3D
            if isinstance(pos, tuple) and handle_pose is not None:
                dist = np.linalg.norm(np.array(handle_pose) - np.array(pos))
                if dist < 0.05 and obj_name != 'handle_pose':  # Handle within 5cm
                    handle_object = obj_name
                    break

        # ---- 5. Try picking the handle (if identified) ----
        if handle_object:
            print(f"[Exploration] Handle object likely named '{handle_object}' at pose {handle_pose}. Attempting pick...")
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=handle_object,
                    target_pos=handle_pose,
                    approach_distance=0.12,
                    max_steps=75,
                    threshold=0.008,
                    approach_axis='z',
                    timeout=12.0
                )
                if done:
                    print("[Exploration] Successfully picked up handle object.")
                else:
                    print("[Exploration] Could not pick handle object.")
            except Exception as e:
                print(f"[Exploration] Exception in execute_pick: {e}")

            # Try pulling (to see if lock status is revealed)
            try:
                print(f"[Exploration] Attempting to pull '{handle_object}' at {handle_pose}...")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer_name='drawer',  # Guessing name; adjust as needed
                    handle_name=handle_object,
                    target_pos=handle_pose,
                    approach_distance=0.12,
                    max_steps=75,
                    threshold=0.008,
                    approach_axis='z',
                    timeout=6.0
                )
                if done:
                    print("[Exploration] Pull action finished (possible drawer unlock/lock-known predicate revealed).")
                else:
                    print("[Exploration] Pull action did not complete.")
            except Exception as e:
                print(f"[Exploration] Exception in execute_pull: {e}")
        else:
            print("[Exploration] Could not find a handle object at handle_pose.")

        # ---- 6. Finalize ---> After the above, check state predicates manually via external checking (not in code).
        print("[Exploration] Exploration phase complete; feedback or logs should indicate which new predicate can now be achieved.")

        # ---- The rest of your code (oracle plan execution) would follow this exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
