# run_skeleton_task.py (Executable exploration code for missing predicate via existing predefined skills)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # DO NOT redefine primitives

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with explicit exploration to find missing predicates.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration for Missing Predicate (robot-at drawer-area) ---
        # Purpose: Ensure that 'robot-at drawer-area' holds, or can be asserted via available skills, since it is missing.

        # The domain uses (robot-at ?p - location) as the predicate for robot position.
        # Feedback says: (robot-at drawer-area) is missing in the state.
        # Our plan: move the robot to the drawer-area and check if that predicate appears.

        # 1. Try to find a "drawer-area" location in the known positions and move robot there.
        # Note: If the position information is not available, this will be a logical placeholder.

        # Try to get drawer-area location (if present), else fallback to any known location key containing "drawer" or "area"
        drawer_area_pos = None
        for k in positions.keys():
            if "drawer-area" in k:
                drawer_area_pos = positions[k]
                drawer_area_name = k
                break
        if drawer_area_pos is None:
            # Fallback: try partial matches for common names
            for k in positions.keys():
                if "drawer" in k or "area" in k:
                    drawer_area_pos = positions[k]
                    drawer_area_name = k
                    break
            else:
                # If still not found, just pick the first location for demonstration
                # In a real task, this should not happen
                for k in positions.keys():
                    drawer_area_pos = positions[k]
                    drawer_area_name = k
                    print("[Warning] Could not find 'drawer-area' in positions, using '%s' for demonstration." % drawer_area_name)
                    break

        print(f"[Exploration] Trying to move robot to {drawer_area_name}: {drawer_area_pos}")

        # At this point, try to move the robot there using a predefined skill
        # Use the skill 'execute_go', which (in the domain) requires from_location and to_location
        try:
            # Assuming we need to determine where the robot currently is.
            # We'll try to infer the robot's current location by looking for a predicate in obs['state'] or descriptions
            # If not found, default to 'start' or first location in positions
            current_loc = None
            robot_state_predicate = None
            # obs or descriptions may have the state information
            for pred in descriptions if descriptions else []:
                if 'robot-at' in pred:
                    # Example: "(robot-at location1)"
                    tokens = pred.replace("(", "").replace(")", "").split()
                    if len(tokens) == 2:
                        current_loc = tokens[1]
                        robot_state_predicate = pred
                        break
            if not current_loc:
                # Fallback: just take the "first" location found, or one with "start" in the name
                for k in positions.keys():
                    if "start" in k:
                        current_loc = k
                        break
                if not current_loc:
                    current_loc = list(positions.keys())[0]
            print(f"[Exploration] Robot moving from {current_loc} to {drawer_area_name}")

            # Execute the move using the available execute_go skill
            obs, reward, done = execute_go(env, task, from_location=current_loc, to_location=drawer_area_name)
            print(f"[Exploration] Executed execute_go from {current_loc} to {drawer_area_name}")
        except Exception as e:
            print(f"[Error] Failed to call execute_go: {e}")

        # Optionally, check if the predicate now appears in the robot's observed state
        # This involves fetching the new observation/state and checking for 'robot-at drawer-area'
        print("[Exploration] Checking for missing predicate '(robot-at drawer-area)' in state...")
        updated_descriptions = None
        try:
            updated_descriptions = task.get_observation()['state']
        except Exception:
            try:
                # Sometimes state info in 'descriptions', fallback to it if needed
                updated_descriptions = descriptions
            except Exception:
                updated_descriptions = None

        found_robot_at_drawer_area = False
        if updated_descriptions:
            for pred in updated_descriptions:
                if 'robot-at' in pred and ('drawer-area' in pred or drawer_area_name in pred):
                    print("[Success] Found predicate in state:", pred)
                    found_robot_at_drawer_area = True
                    break
        if not found_robot_at_drawer_area:
            print("[Notice] Predicate '(robot-at drawer-area)' still missing after exploration.")
            print("[Info] This may hint at a modeling error or a missing step in the plan that must be handled.")

        # === Now Continue with Oracle Plan as needed ===
        # After confirming/investigating the missing predicate, the oracle plan can continue.
        # For demonstration, insert further planned actions here.
        # For example:
        # obs, reward, done = execute_pick(env, task, object_name, location_name)
        # obs, reward, done = execute_place(env, task, object_name, drawer_name, location_name)
        # (plan continues...)

        # For this exploration code, the focus is solely on identifying predicate issues.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
