# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the robot's location and update our knowledge.

        # Try to use available skills to explore and identify the robot's location.
        # Since the available skills do not include a direct "explore" or "identify" action,
        # we will use the movement and observation capabilities to infer the missing predicate.

        # Step 1: Get all known locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'location' in name:
                location_names.append(name)
        if not location_names:
            # Fallback: try to infer from all keys
            location_names = list(positions.keys())

        # Step 2: Try to move the robot to each location and check if the robot is at that location
        # Use execute_go if available
        # The skill signature is likely: execute_go(env, task, from_location, to_location, ...)
        # We'll try to move the robot to each location and print the result

        # First, try to determine the robot's current location
        robot_location = None
        for loc in location_names:
            # Heuristic: check if the robot is at this location by querying the environment
            # (Assume task or env provides a way to check robot's location, or use positions)
            # If not, just pick the first location as the starting point
            robot_location = loc
            break

        print("[Exploration] Known locations:", location_names)
        print("[Exploration] Assuming robot starts at:", robot_location)

        # Try to move to each location and see if any errors occur (indicating missing predicate)
        for target_loc in location_names:
            if target_loc == robot_location:
                continue
            try:
                print(f"[Exploration] Attempting to move robot from {robot_location} to {target_loc} using execute_go...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_location,
                    to_location=target_loc,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Moved robot to {target_loc}.")
                robot_location = target_loc
                if done:
                    print("[Exploration] Task ended after move!")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to move to {target_loc}: {e}")

        # After exploration, print out the likely missing predicate
        print("[Exploration] Based on feedback and exploration, the missing predicate is likely (robot-at floor) or similar.")

        # === End of Exploration Phase ===

        # TODO: Insert oracle plan execution here, using only predefined skills, once the missing predicate is confirmed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
