# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, specifically (robot-at floor).
        # We will use available skills to explore and confirm the robot's location predicates.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use 'execute_go' to move the robot and observe the effect on predicates.
        # This will help us determine if (robot-at floor) is a valid predicate and if the skill works as expected.

        # For demonstration, let's assume the environment has at least two locations: 'floor' and another (e.g., 'table' or 'drawer_area').
        # We'll try to move the robot from 'floor' to another location and back, using only the available skills.

        # Find all locations from positions (assuming keys like 'floor', 'drawer_area', etc.)
        location_names = [name for name in positions if 'location' in name or 'floor' in name or 'area' in name or 'room' in name]
        if 'floor' not in location_names:
            location_names.append('floor')  # Ensure 'floor' is present for exploration

        # For exploration, pick two locations
        if len(location_names) < 2:
            print("[Exploration] Not enough locations found for exploration. Please check object_positions.")
            return

        from_location = location_names[0]
        to_location = location_names[1]

        print(f"[Exploration] Attempting to move robot from '{from_location}' to '{to_location}' using execute_go.")

        try:
            # Use execute_go to move the robot
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] Robot moved from '{from_location}' to '{to_location}'.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go: {e}")

        # Now, move back to 'floor' to check if (robot-at floor) is a valid predicate
        print(f"[Exploration] Attempting to move robot back to 'floor' using execute_go.")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                to_location,
                'floor'
            )
            print(f"[Exploration] Robot moved back to 'floor'.")
        except Exception as e:
            print(f"[Exploration] Error during execute_go back to 'floor': {e}")

        # At this point, we have explored the robot's location predicates using only available skills.
        # If the above actions succeed, (robot-at floor) is a valid predicate and can be used in planning.
        # If not, further investigation of the domain and observation is needed.

        print("[Exploration] Exploration phase complete. Check logs for predicate effects and missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()