# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the environment to find which predicate is missing.
        # We will attempt to use available skills to interact with objects and observe the effects.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For demonstration, we will attempt to use each skill in a safe, exploratory way.
        # This will help us identify if any predicate is missing or if an action fails due to a missing precondition.

        # Get all objects and locations from positions
        object_names = [name for name in positions if 'object' in name or 'obj' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'drawer' in name]
        drawer_names = [name for name in positions if 'drawer' in name]
        switch_names = [name for name in positions if 'switch' in name]

        # Try to turn on the light if possible (execute_push_switch is not in available_skills, so skip)
        # Instead, try to use execute_go to move to a location and see if any action is blocked by room-dark

        # 1. Try to move to another location (if more than one location exists)
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Trying to move from {from_loc} to {to_loc} using execute_go.")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print("[Exploration] execute_go succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick up an object (if any)
        if object_names and location_names:
            obj = object_names[0]
            loc = location_names[0]
            try:
                print(f"[Exploration] Trying to pick up {obj} at {loc} using execute_pick.")
                obs, reward, done = execute_pick(env, task, obj, loc)
                print("[Exploration] execute_pick succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to open a drawer (if any drawer and handle object exist)
        if drawer_names and object_names and location_names:
            drawer = drawer_names[0]
            handle = object_names[0]
            loc = location_names[0]
            try:
                print(f"[Exploration] Trying to pull {drawer} with handle {handle} at {loc} using execute_pull.")
                obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                print("[Exploration] execute_pull succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try to place an object in a drawer (if possible)
        if object_names and drawer_names and location_names:
            obj = object_names[0]
            drawer = drawer_names[0]
            loc = location_names[0]
            try:
                print(f"[Exploration] Trying to place {obj} in {drawer} at {loc} using execute_place.")
                obs, reward, done = execute_place(env, task, obj, drawer, loc)
                print("[Exploration] execute_place succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_place failed: {e}")

        # 5. Try to push a drawer closed (if any)
        if drawer_names and location_names:
            drawer = drawer_names[0]
            loc = location_names[0]
            try:
                print(f"[Exploration] Trying to push {drawer} closed at {loc} using execute_push.")
                obs, reward, done = execute_push(env, task, drawer, loc)
                print("[Exploration] execute_push succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to sweep an object (if any)
        if object_names and location_names:
            obj = object_names[0]
            loc = location_names[0]
            try:
                print(f"[Exploration] Trying to sweep {obj} at {loc} using execute_sweep.")
                obs, reward, done = execute_sweep(env, task, obj, loc)
                print("[Exploration] execute_sweep succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try to use the gripper (if possible)
        try:
            print(f"[Exploration] Trying to use gripper using execute_gripper.")
            obs, reward, done = execute_gripper(env, task)
            print("[Exploration] execute_gripper succeeded.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # 8. Try to rotate (if available)
        if 'execute_rotate' in available_skills:
            try:
                print(f"[Exploration] Trying to rotate using execute_rotate.")
                obs, reward, done = execute_rotate(env, task)
                print("[Exploration] execute_rotate succeeded.")
            except Exception as e:
                print(f"[Exploration] execute_rotate failed: {e}")

        print("[Exploration] Exploration phase complete. Check above for any failed actions or missing predicates.")

        # === End of Exploration Phase ===

        # TODO: After exploration, use the feedback to update your plan or report the missing predicate.
        # For now, the code above will help you identify which predicate is missing by observing which action fails and why.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()