# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try each available skill and observe the environment for changes,
        # to infer which predicate is missing or required for the plan to proceed.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we will attempt to execute each skill in a safe, generic way.
        # We will log the result and any exceptions to help identify missing predicates.

        print("=== Exploration Phase: Trying available skills to identify missing predicate ===")
        for skill_name in available_skills:
            skill_func = globals().get(skill_name, None)
            if skill_func is None:
                print(f"[Exploration] Skill {skill_name} not found in skill_code.")
                continue

            print(f"[Exploration] Attempting skill: {skill_name}")
            try:
                # Prepare generic arguments for each skill
                # These are placeholders; in a real scenario, you would extract these from the observation or positions
                # For demonstration, we use the first available object/location/drawer as needed

                # Get available objects, drawers, locations from positions
                object_names = [k for k in positions.keys() if 'object' in k or 'obj' in k]
                drawer_names = [k for k in positions.keys() if 'drawer' in k]
                location_names = [k for k in positions.keys() if 'room' in k or 'loc' in k or 'area' in k]

                # Default arguments
                obj = object_names[0] if object_names else None
                drawer = drawer_names[0] if drawer_names else None
                loc1 = location_names[0] if location_names else None
                loc2 = location_names[1] if len(location_names) > 1 else loc1

                # Call each skill with plausible arguments
                if skill_name == 'execute_pick':
                    if obj and loc1:
                        obs, reward, done = skill_func(env, task, obj, loc1)
                        print(f"[Exploration] {skill_name}({obj}, {loc1}) executed. Done: {done}")
                elif skill_name == 'execute_place':
                    if obj and drawer and loc1:
                        obs, reward, done = skill_func(env, task, obj, drawer, loc1)
                        print(f"[Exploration] {skill_name}({obj}, {drawer}, {loc1}) executed. Done: {done}")
                elif skill_name == 'execute_push':
                    if drawer and loc1:
                        obs, reward, done = skill_func(env, task, drawer, loc1)
                        print(f"[Exploration] {skill_name}({drawer}, {loc1}) executed. Done: {done}")
                elif skill_name == 'execute_pull':
                    # For pull, need a handle object and a drawer
                    handle = None
                    for o in object_names:
                        if 'handle' in o:
                            handle = o
                            break
                    if drawer and handle and loc1:
                        obs, reward, done = skill_func(env, task, drawer, handle, loc1)
                        print(f"[Exploration] {skill_name}({drawer}, {handle}, {loc1}) executed. Done: {done}")
                elif skill_name == 'execute_sweep':
                    if obj and loc1:
                        obs, reward, done = skill_func(env, task, obj, loc1)
                        print(f"[Exploration] {skill_name}({obj}, {loc1}) executed. Done: {done}")
                elif skill_name == 'execute_rotate':
                    # If rotate needs an object and location
                    if obj and loc1:
                        obs, reward, done = skill_func(env, task, obj, loc1)
                        print(f"[Exploration] {skill_name}({obj}, {loc1}) executed. Done: {done}")
                elif skill_name == 'execute_go':
                    if loc1 and loc2:
                        obs, reward, done = skill_func(env, task, loc1, loc2)
                        print(f"[Exploration] {skill_name}({loc1}, {loc2}) executed. Done: {done}")
                elif skill_name == 'execute_gripper':
                    obs, reward, done = skill_func(env, task)
                    print(f"[Exploration] {skill_name}() executed. Done: {done}")
                else:
                    print(f"[Exploration] No argument template for {skill_name}, skipping.")
            except Exception as e:
                print(f"[Exploration] Exception during {skill_name}: {e}")

        print("=== Exploration Phase Complete ===")
        print("Check the above logs and environment state to infer which predicate is missing or required for planning.")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
