# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We will attempt to use available exploration skills to discover missing predicates.
        # The exploration domain provides skills like execute_go_identify, execute_go_temperature, etc.
        # However, only the following skills are available in this environment:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper', 'execute_push_swtich -> push']

        # Since the exploration skills are not directly available, we will use the available skills to interact with the environment
        # and observe if any precondition failures or unexpected states occur, which may indicate missing predicates.

        # For demonstration, let's attempt to perform each available skill in a safe, exploratory manner.
        # We will log any exceptions or failures to help identify missing predicates.

        available_skills = [
            'execute_pick',
            'execute_place',
            'execute_push',
            'execute_pull',
            'execute_sweep',
            'execute_rotate',
            'execute_go',
            'execute_gripper',
            'execute_push_switch'
        ]

        # Map skill names to actual functions (assuming skill_code provides them)
        skill_func_map = {
            'execute_pick': execute_pick if 'execute_pick' in globals() else None,
            'execute_place': execute_place if 'execute_place' in globals() else None,
            'execute_push': execute_push if 'execute_push' in globals() else None,
            'execute_pull': execute_pull if 'execute_pull' in globals() else None,
            'execute_sweep': execute_sweep if 'execute_sweep' in globals() else None,
            'execute_rotate': execute_rotate if 'execute_rotate' in globals() else None,
            'execute_go': execute_go if 'execute_go' in globals() else None,
            'execute_gripper': execute_gripper if 'execute_gripper' in globals() else None,
            'execute_push_switch': execute_push_switch if 'execute_push_switch' in globals() else None,
        }

        # Get a list of objects, drawers, locations, switches from positions or descriptions
        # This is a placeholder; in practice, you would parse these from the observation or environment
        objects = []
        drawers = []
        locations = []
        switches = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'switch' in name:
                switches.append(name)
            elif 'loc' in name or 'room' in name:
                locations.append(name)
            else:
                objects.append(name)

        # If no locations found, use some default
        if not locations:
            locations = ['location1', 'location2']

        # If no drawers or switches, use placeholders
        if not drawers:
            drawers = ['drawer1']
        if not switches:
            switches = ['switch1']

        # If no objects, use placeholder
        if not objects:
            objects = ['object1']

        # Set initial robot location
        robot_location = locations[0]

        # === Exploration: Try each skill and catch errors ===
        print("=== Exploration Phase: Trying available skills to detect missing predicates ===")
        for skill_name in available_skills:
            func = skill_func_map.get(skill_name)
            if func is None:
                print(f"[Exploration] Skill {skill_name} not implemented in skill_code.")
                continue
            try:
                print(f"[Exploration] Attempting skill: {skill_name}")
                # Call each skill with plausible arguments
                if skill_name == 'execute_pick':
                    # Try to pick the first object at the first location
                    obs, reward, done = func(
                        env,
                        task,
                        obj=objects[0],
                        location=robot_location
                    )
                elif skill_name == 'execute_place':
                    # Try to place the first object into the first drawer at the first location
                    obs, reward, done = func(
                        env,
                        task,
                        obj=objects[0],
                        drawer=drawers[0],
                        location=robot_location
                    )
                elif skill_name == 'execute_push':
                    # Try to push the first drawer at the first location
                    obs, reward, done = func(
                        env,
                        task,
                        drawer=drawers[0],
                        location=robot_location
                    )
                elif skill_name == 'execute_pull':
                    # Try to pull the first drawer handle at the first location
                    obs, reward, done = func(
                        env,
                        task,
                        drawer=drawers[0],
                        handle=objects[0],
                        location=robot_location
                    )
                elif skill_name == 'execute_sweep':
                    # Try to sweep the first object at the first location
                    obs, reward, done = func(
                        env,
                        task,
                        obj=objects[0],
                        location=robot_location
                    )
                elif skill_name == 'execute_rotate':
                    # Try to rotate (if applicable)
                    obs, reward, done = func(
                        env,
                        task
                    )
                elif skill_name == 'execute_go':
                    # Try to move from location 0 to location 1
                    if len(locations) > 1:
                        obs, reward, done = func(
                            env,
                            task,
                            from_location=locations[0],
                            to_location=locations[1]
                        )
                    else:
                        print("[Exploration] Not enough locations to execute_go.")
                        continue
                elif skill_name == 'execute_gripper':
                    # Try to use gripper
                    obs, reward, done = func(
                        env,
                        task
                    )
                elif skill_name == 'execute_push_switch':
                    # Try to push the first switch at the first location
                    obs, reward, done = func(
                        env,
                        task,
                        switch=switches[0],
                        location=robot_location
                    )
                else:
                    print(f"[Exploration] No argument template for skill {skill_name}.")
                    continue
                print(f"[Exploration] Skill {skill_name} executed successfully.")
            except Exception as e:
                print(f"[Exploration] Skill {skill_name} failed with error: {e}")
                # Log the error for further analysis; this may indicate a missing predicate or precondition

        print("=== Exploration Phase Complete ===")
        print("Check above for any failed skills and error messages indicating missing predicates.")

        # === End of Exploration ===
        # At this point, based on the errors encountered, you can analyze which predicate or precondition is missing.
        # This information can be used to update the domain/problem PDDL or to inform further debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
