# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and check if the robot can be at 'floor' or similar locations.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use 'execute_go' to move the robot to different locations and observe the result.
        # If the robot cannot be at 'floor', this will be revealed by the environment or by errors.

        # For demonstration, let's try to move the robot to all known locations in positions.
        # We assume 'positions' is a dict mapping object/location names to their positions.

        # Find all location names (assuming keys that are not objects)
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'drawer' in name or 'location' in name:
                location_names.append(name)
        # If no location found, fallback to all keys
        if not location_names:
            location_names = list(positions.keys())

        # Try to move the robot to each location using execute_go
        # We need to know the robot's current location; let's assume it's the first location
        if not location_names:
            print("[Exploration] No locations found in object positions.")
        else:
            current_loc = location_names[0]
            for loc in location_names:
                if loc == current_loc:
                    continue
                print(f"[Exploration] Attempting to move robot from {current_loc} to {loc} using execute_go.")
                try:
                    obs, reward, done = execute_go(
                        env,
                        task,
                        from_location=current_loc,
                        to_location=loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Robot moved to {loc}.")
                    current_loc = loc
                    if done:
                        print("[Exploration] Task ended during exploration.")
                        break
                except Exception as e:
                    print(f"[Exploration] Failed to move to {loc}: {e}")

        # After exploration, you can check the robot's state or the environment's feedback to see if (robot-at floor) is now true or if the predicate is missing.

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after identifying the missing predicate.
        # For now, the code focuses on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
