# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discovering Missing Predicate ===
        # Feedback: (holding handle_bottom)
        # The robot is holding 'handle_bottom', but the plan fails due to a missing predicate.
        # We need to explore which predicate is missing by using available skills and observing the environment.

        # Step 1: Identify relevant objects and locations
        # For this example, we assume 'handle_bottom' is the handle object, and there is a drawer (e.g., 'drawer_1')
        # and a location (e.g., 'location_1'). These names should match your environment's object naming.
        handle_name = 'handle_bottom'
        drawer_name = 'drawer_1'
        robot_location = None
        handle_location = None

        # Try to get the robot's and handle's positions from the environment
        try:
            handle_location = positions[handle_name]
        except KeyError:
            print(f"[Exploration] Handle object '{handle_name}' not found in positions.")
        # Find robot's current location (if available)
        for key in positions:
            if 'robot' in key:
                robot_location = positions[key]
                break

        # If not found, use default names (these should be adapted to your environment)
        if robot_location is None:
            robot_location = 'location_1'
        if handle_location is None:
            handle_location = 'location_2'

        # Step 2: Try to pick the handle (execute_pick)
        print(f"[Exploration] Attempting to pick '{handle_name}' at location '{handle_location}'")
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                obj=handle_name,
                p=handle_location
            )
            if done:
                print(f"[Exploration] Successfully picked '{handle_name}'.")
            else:
                print(f"[Exploration] Pick action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_pick: {e}")

        # Step 3: Try to pull the drawer using the handle (execute_pull)
        print(f"[Exploration] Attempting to pull '{drawer_name}' using '{handle_name}' at location '{handle_location}'")
        try:
            obs, reward, done = execute_pull(
                env,
                task,
                d=drawer_name,
                h=handle_name,
                p=handle_location
            )
            if done:
                print(f"[Exploration] Successfully pulled '{drawer_name}' with '{handle_name}'.")
            else:
                print(f"[Exploration] Pull action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_pull: {e}")

        # Step 4: If the pull fails, try to push (execute_push) in case the drawer is already open
        print(f"[Exploration] Attempting to push '{drawer_name}' at location '{handle_location}'")
        try:
            obs, reward, done = execute_push(
                env,
                task,
                d=drawer_name,
                p=handle_location
            )
            if done:
                print(f"[Exploration] Successfully pushed '{drawer_name}'.")
            else:
                print(f"[Exploration] Push action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_push: {e}")

        # Step 5: Try to move the robot to another location (execute_go)
        # This can help in identifying if the robot's location is a missing predicate
        another_location = 'location_3'
        print(f"[Exploration] Attempting to move robot from '{robot_location}' to '{another_location}'")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_loc=robot_location,
                to=another_location
            )
            if done:
                print(f"[Exploration] Successfully moved robot to '{another_location}'.")
            else:
                print(f"[Exploration] Move action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # Step 6: Try to place an object in the drawer (execute_place)
        # This will help check if predicates like 'drawer-open' or 'drawer-unlocked' are missing
        print(f"[Exploration] Attempting to place '{handle_name}' in '{drawer_name}' at location '{handle_location}'")
        try:
            obs, reward, done = execute_place(
                env,
                task,
                o=handle_name,
                d=drawer_name,
                p=handle_location
            )
            if done:
                print(f"[Exploration] Successfully placed '{handle_name}' in '{drawer_name}'.")
            else:
                print(f"[Exploration] Place action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_place: {e}")

        # Step 7: Try to use other available skills for further exploration
        # For example, execute_sweep, execute_gripper, execute_rotate (if implemented)
        print(f"[Exploration] Attempting to sweep '{handle_name}' at location '{handle_location}'")
        try:
            obs, reward, done = execute_sweep(
                env,
                task,
                o=handle_name,
                p=handle_location
            )
            if done:
                print(f"[Exploration] Successfully swept '{handle_name}'.")
            else:
                print(f"[Exploration] Sweep action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_sweep: {e}")

        print(f"[Exploration] Attempting to use gripper")
        try:
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print(f"[Exploration] Successfully used gripper.")
            else:
                print(f"[Exploration] Gripper action did not complete as expected.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_gripper: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Review logs to identify missing predicates or preconditions.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()