# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate.
        # We use the exploration domain knowledge to try to discover which predicate is missing.
        # This is done by attempting the exploration skills and observing the environment's response.

        # For demonstration, we attempt to identify, weigh, and check durability of all objects at all locations.
        # This will help us discover if any of the predicates (identified, weight-known, durability-known, lock-known, temperature-known)
        # are missing or required for the main plan.

        # Assume positions is a dict: {object_name: (x, y, z, location_name)}
        # We'll collect all unique objects and locations.
        object_names = []
        location_names = set()
        for obj, pos in positions.items():
            object_names.append(obj)
            if len(pos) == 4:
                location_names.add(pos[3])
        object_names = list(set(object_names))
        location_names = list(location_names)

        # Try to identify all objects at all locations
        for loc in location_names:
            try:
                # Move robot to location (if not already there)
                # Use execute_go if available
                # For exploration, we use execute_go_identify, execute_go_temperature, etc.
                # But only use predefined skills; so we use execute_go if available
                if 'execute_go' in globals():
                    # Find current robot location (assume first location if unknown)
                    current_loc = location_names[0]
                    if current_loc != loc:
                        print(f"[Exploration] Moving robot from {current_loc} to {loc} for identification.")
                        try:
                            obs, reward, done = execute_go(env, task, current_loc, loc)
                        except Exception as e:
                            print(f"[Exploration] execute_go failed: {e}")
                        current_loc = loc
                # Try to pick each object at this location to trigger weight-known/durability-known
                for obj in object_names:
                    obj_pos = positions.get(obj, None)
                    if obj_pos is not None and len(obj_pos) == 4 and obj_pos[3] == loc:
                        # Try to pick for weight-known
                        if 'execute_pick' in globals():
                            print(f"[Exploration] Trying to pick {obj} at {loc} to check for missing predicates.")
                            try:
                                obs, reward, done = execute_pick(env, task, obj, loc)
                            except Exception as e:
                                print(f"[Exploration] execute_pick failed: {e}")
                        # Try to sweep (if available)
                        if 'execute_sweep' in globals():
                            print(f"[Exploration] Trying to sweep {obj} at {loc}.")
                            try:
                                obs, reward, done = execute_sweep(env, task, obj, loc)
                            except Exception as e:
                                print(f"[Exploration] execute_sweep failed: {e}")
            except Exception as e:
                print(f"[Exploration] Error during exploration at {loc}: {e}")

        # Try to pull handles to check for lock-known predicate
        for obj in object_names:
            # If the object is a handle (we don't know, so try all)
            for loc in location_names:
                if 'execute_pull' in globals():
                    print(f"[Exploration] Trying to pull {obj} at {loc} to check for lock-known predicate.")
                    try:
                        obs, reward, done = execute_pull(env, task, obj, obj, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")

        # Try to push switches to check for room-bright/room-dark predicates
        for loc in location_names:
            if 'execute_push_switch' in globals():
                print(f"[Exploration] Trying to push switch at {loc} to check for room-bright predicate.")
                try:
                    obs, reward, done = execute_push_switch(env, task, "switch", loc)
                except Exception as e:
                    print(f"[Exploration] execute_push_switch failed: {e}")

        # === End of Exploration Phase ===

        # === Main Task Plan Execution ===
        # After exploration, proceed with the main plan using only predefined skills.
        # Example: Pick up an object and place it in a drawer.

        # For demonstration, pick the first object and first location/drawer found.
        if object_names and location_names:
            obj = object_names[0]
            loc = location_names[0]
            # Try to turn on the light if needed
            if 'execute_push_switch' in globals():
                print(f"[Task] Turning on the light at {loc}.")
                try:
                    obs, reward, done = execute_push_switch(env, task, "switch", loc)
                except Exception as e:
                    print(f"[Task] execute_push_switch failed: {e}")
            # Try to pick the object
            if 'execute_pick' in globals():
                print(f"[Task] Picking up {obj} at {loc}.")
                try:
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Task] execute_pick failed: {e}")
            # Try to place the object in a drawer (if any drawer exists)
            drawer_name = None
            for name in object_names:
                if "drawer" in name:
                    drawer_name = name
                    break
            if drawer_name and 'execute_place' in globals():
                print(f"[Task] Placing {obj} in {drawer_name} at {loc}.")
                try:
                    obs, reward, done = execute_place(env, task, obj, drawer_name, loc)
                except Exception as e:
                    print(f"[Task] execute_place failed: {e}")
            # Try to close the drawer
            if drawer_name and 'execute_push' in globals():
                print(f"[Task] Closing {drawer_name} at {loc}.")
                try:
                    obs, reward, done = execute_push(env, task, drawer_name, loc)
                except Exception as e:
                    print(f"[Task] execute_push failed: {e}")

        print("[Task] Main plan execution complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
