# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the presence/absence of this predicate.

        # For exploration, we will use available skills to move the robot and observe the state.
        # Since only predefined skills can be used, we will use 'execute_go' to move the robot.

        # Let's try to move the robot to 'floor' (if such a location exists) and check the result.
        # We assume 'positions' contains location names.

        # Find all locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'location' in name or 'floor' in name or 'room' in name:
                location_names.append(name)
        # If no explicit location names, fallback to all keys
        if not location_names:
            location_names = list(positions.keys())

        # Try to find 'floor' location
        floor_location = None
        for loc in location_names:
            if 'floor' in loc:
                floor_location = loc
                break

        # If not found, just pick the first location as a fallback
        if floor_location is None and location_names:
            floor_location = location_names[0]

        # Get current robot location from observation or positions
        current_robot_location = None
        for name in positions:
            if 'robot' in name or 'robby' in name:
                current_robot_location = positions[name]
                break

        # If not found, just use the first location
        if current_robot_location is None and location_names:
            current_robot_location = location_names[0]

        # Now, try to move the robot to 'floor' using execute_go
        # The skill signature is likely: execute_go(env, task, from_location, to_location, ...)
        # We'll use try/except to handle any errors gracefully

        print("[Exploration] Attempting to move robot to 'floor' location to check for missing predicate...")

        try:
            # If current_robot_location and floor_location are both names, use them directly
            obs, reward, done = execute_go(
                env,
                task,
                from_location=current_robot_location,
                to_location=floor_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print(f"[Exploration] Robot moved from {current_robot_location} to {floor_location}.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to 'floor': {e}")

        # After moving, check if the predicate (robot-at floor) is now true in the environment
        # This would typically be done by querying the environment state or observation
        # For demonstration, print the current observation
        try:
            current_obs = task.get_observation()
            print("[Exploration] Current observation after moving to 'floor':", current_obs)
        except Exception as e:
            print(f"[Exploration] Could not retrieve observation: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan after exploration, using only predefined skills.
        # For now, the focus is on identifying the missing predicate as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
