# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to check for the (room-bright) predicate.
        # We'll attempt to execute actions that require (room-bright) and handle failures.
        # If actions fail due to missing (room-bright), we will try to turn on the light.

        # For demonstration, let's assume the following:
        # - There is a switch object and its location.
        # - There is at least one object to pick up from the floor.
        # - There is a drawer and its handle.
        # - The robot starts at some initial location.

        # You may need to adjust these keys based on your environment's object naming.
        try:
            # Example object keys (replace with actual keys from your environment)
            switch_name = None
            switch_location = None
            for obj_name, pos in positions.items():
                if "switch" in obj_name:
                    switch_name = obj_name
                    switch_location = pos
                    break

            # Find an object on the floor to pick
            object_to_pick = None
            object_to_pick_location = None
            for obj_name, pos in positions.items():
                if "object" in obj_name or "ball" in obj_name:
                    object_to_pick = obj_name
                    object_to_pick_location = pos
                    break

            # Find a drawer and its handle
            drawer_name = None
            drawer_location = None
            handle_name = None
            for obj_name, pos in positions.items():
                if "drawer" in obj_name:
                    drawer_name = obj_name
                    drawer_location = pos
                if "handle" in obj_name:
                    handle_name = obj_name

            # Assume robot's initial location is known
            robot_location = None
            for obj_name, pos in positions.items():
                if "robot" in obj_name:
                    robot_location = pos
                    break

            # If not found, set to some default or first location
            if robot_location is None:
                robot_location = list(positions.values())[0]

            # === Try to pick up an object (requires room-bright) ===
            print("[Exploration] Attempting to pick up object:", object_to_pick)
            try:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=object_to_pick,
                    p=object_to_pick_location
                )
                print("[Exploration] Pick succeeded. (room-bright) likely true.")
            except Exception as e:
                print("[Exploration] Pick failed. Exception:", e)
                print("[Exploration] (room-bright) may be missing. Attempting to turn on the light.")

                # Try to turn on the light (execute_push_switch)
                try:
                    obs, reward, done = execute_push_switch(
                        env,
                        task,
                        s=switch_name,
                        p=switch_location
                    )
                    print("[Exploration] Light switch pressed. (room-bright) should now be true.")

                    # Try picking up the object again
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=object_to_pick,
                        p=object_to_pick_location
                    )
                    print("[Exploration] Pick succeeded after turning on the light.")
                except Exception as e2:
                    print("[Exploration] Failed to turn on the light or pick object. Exception:", e2)
                    print("[Exploration] Exploration failed. Cannot proceed.")
                    return

            # === Continue with the rest of the plan (example: place object in drawer) ===
            # Open the drawer (requires holding the handle, drawer unlocked, etc.)
            if handle_name and drawer_name and drawer_location:
                try:
                    # Pick up the handle first
                    print("[Task] Picking up handle:", handle_name)
                    obs, reward, done = execute_pick(
                        env,
                        task,
                        obj=handle_name,
                        p=drawer_location
                    )
                    print("[Task] Handle picked.")

                    # Pull the drawer open
                    print("[Task] Pulling drawer open:", drawer_name)
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        d=drawer_name,
                        h=handle_name,
                        p=drawer_location
                    )
                    print("[Task] Drawer opened.")
                except Exception as e:
                    print("[Task] Failed to open drawer. Exception:", e)

            # Place the object in the drawer
            if drawer_name and object_to_pick:
                try:
                    print("[Task] Placing object in drawer.")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        o=object_to_pick,
                        d=drawer_name,
                        p=drawer_location
                    )
                    print("[Task] Object placed in drawer.")
                except Exception as e:
                    print("[Task] Failed to place object in drawer. Exception:", e)

            # Optionally, close the drawer
            if drawer_name:
                try:
                    print("[Task] Closing drawer.")
                    obs, reward, done = execute_push(
                        env,
                        task,
                        d=drawer_name,
                        p=drawer_location
                    )
                    print("[Task] Drawer closed.")
                except Exception as e:
                    print("[Task] Failed to close drawer. Exception:", e)

        except Exception as main_e:
            print("[Main] Exception during exploration or task execution:", main_e)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()