# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We need to explore the environment to confirm the robot's location and update our knowledge.

        # Try to use available skills to explore and confirm the robot's location.
        # Since we do not have a dedicated exploration skill, we use the available skills to move and observe.

        # Step 1: Get all known locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'location' in name:
                location_names.append(name)
        # If no explicit location names, fallback to all keys
        if not location_names:
            location_names = list(positions.keys())

        # Step 2: Try to move the robot to each location using execute_go
        # and check if the robot is at 'floor' (as per feedback)
        robot_current_location = None
        for loc in location_names:
            try:
                # Try to move to this location using execute_go
                # We need a 'from' and 'to' location; if robot_current_location is None, assume first location is current
                if robot_current_location is None:
                    robot_current_location = loc
                    continue  # Skip moving to the same location
                print(f"[Exploration] Moving robot from {robot_current_location} to {loc}")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=robot_current_location,
                    to_location=loc,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                robot_current_location = loc
                # After moving, check if the robot is at 'floor'
                if 'floor' in loc:
                    print("[Exploration] Robot is now at 'floor'. Predicate (robot-at floor) is confirmed.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to move to {loc}: {e}")
                continue

        # Step 3: If not found, try to use execute_sweep or other skills to further explore
        if robot_current_location is not None and 'floor' not in robot_current_location:
            for loc in location_names:
                if 'floor' in loc:
                    try:
                        print(f"[Exploration] Attempting to sweep at {loc} to confirm presence.")
                        obs, reward, done = execute_sweep(
                            env,
                            task,
                            object_name=None,  # No specific object, just sweep
                            location=loc,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        print("[Exploration] Sweep executed at floor.")
                        break
                    except Exception as e:
                        print(f"[Exploration] Failed to sweep at {loc}: {e}")
                        continue

        print("[Exploration] Exploration phase complete. Missing predicate likely (robot-at floor).")

        # === End of Exploration Phase ===

        # TODO: Continue with the main oracle plan using the confirmed predicates and available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
