# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and check if the robot can be at 'floor'
        # and if any action fails due to a missing predicate.

        # For demonstration, let's try to move the robot to 'floor' using execute_go
        # and observe if any error or feedback is produced.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use 'execute_go' to move the robot to 'floor'
        # and see if the predicate (robot-at floor) is handled.

        # First, determine the robot's current location and the 'floor' location
        # For this, we assume 'positions' contains keys like 'robot', 'floor', etc.

        robot_location = None
        floor_location = None

        # Try to infer robot's current location and the floor location from positions
        for key in positions:
            if 'robot' in key:
                robot_location = key
            if 'floor' in key:
                floor_location = key

        # If not found, use default names
        if robot_location is None:
            robot_location = 'robot'
        if floor_location is None:
            floor_location = 'floor'

        print(f"[Exploration] Robot location: {robot_location}, Floor location: {floor_location}")

        # Try to move the robot to the floor using execute_go
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {floor_location} using execute_go...")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=floor_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] execute_go completed. Observation: {obs}, Reward: {reward}, Done: {done}")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")
            print("[Exploration] This may indicate a missing predicate such as (robot-at floor) in the initial state or domain.")

        # If the above fails, try to use another skill to interact with the floor
        # For example, try to pick an object on the floor if such an object exists
        object_on_floor = None
        for obj_key in positions:
            if 'object' in obj_key or 'obj' in obj_key:
                object_on_floor = obj_key
                break

        if object_on_floor is not None:
            try:
                print(f"[Exploration] Attempting to pick {object_on_floor} at {floor_location} using execute_pick...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    object_name=object_on_floor,
                    location_name=floor_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] execute_pick completed. Observation: {obs}, Reward: {reward}, Done: {done}")
            except Exception as e:
                print(f"[Exploration] Exception during execute_pick: {e}")
                print("[Exploration] This may further indicate a missing predicate or a problem with the initial state.")

        # The above exploration attempts to surface missing predicates by using available skills
        # and observing any errors or feedback from the environment.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()