# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and try to discover missing predicates.
        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use each skill in a safe, exploratory way and observe the environment's response.
        # This will help us infer which predicate or precondition is missing.

        # For demonstration, we will:
        # 1. Try to move (execute_go) between locations if possible.
        # 2. Try to pick up objects (execute_pick).
        # 3. Try to pull handles (execute_pull).
        # 4. Try to place objects (execute_place).
        # 5. Try to push drawers (execute_push).
        # 6. Try to sweep (execute_sweep).
        # 7. Try to use gripper (execute_gripper).
        # 8. Try to rotate (execute_rotate) if available.

        # The actual arguments for each skill depend on the environment.
        # We'll use the positions dictionary and try to infer object/location names.

        # --- Step 1: List all objects, drawers, locations, switches ---
        # For this example, we assume positions contains keys like:
        #   'objects': list of object names
        #   'drawers': list of drawer names
        #   'locations': list of location names
        #   'switches': list of switch names
        # If not, we try to infer from the keys.

        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        switches = positions.get('switches', [])

        # Fallback: try to infer from all keys
        if not objects or not drawers or not locations:
            all_keys = list(positions.keys())
            # Heuristically assign
            objects = [k for k in all_keys if 'object' in k or 'obj' in k]
            drawers = [k for k in all_keys if 'drawer' in k]
            locations = [k for k in all_keys if 'room' in k or 'loc' in k]
            switches = [k for k in all_keys if 'switch' in k]

        # If still empty, just use all keys as possible objects/locations
        if not objects:
            objects = list(positions.keys())
        if not locations:
            locations = list(positions.keys())

        # --- Step 2: Try to move between locations ---
        print("[Exploration] Trying to move between locations...")
        for i in range(len(locations)):
            for j in range(len(locations)):
                if i != j:
                    from_loc = locations[i]
                    to_loc = locations[j]
                    try:
                        print(f"[Exploration] execute_go: {from_loc} -> {to_loc}")
                        obs, reward, done = execute_go(env, task, from_loc, to_loc)
                        if done:
                            print(f"[Exploration] Task ended after execute_go from {from_loc} to {to_loc}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_go failed: {e}")

        # --- Step 3: Try to pick up each object at each location ---
        print("[Exploration] Trying to pick up objects...")
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] execute_pick: {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after execute_pick {obj} at {loc}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # --- Step 4: Try to pull each drawer handle at each location ---
        print("[Exploration] Trying to pull handles...")
        for drawer in drawers:
            for obj in objects:
                for loc in locations:
                    try:
                        print(f"[Exploration] execute_pull: {drawer}, handle {obj} at {loc}")
                        obs, reward, done = execute_pull(env, task, drawer, obj, loc)
                        if done:
                            print(f"[Exploration] Task ended after execute_pull {drawer}, handle {obj} at {loc}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed: {e}")

        # --- Step 5: Try to place each object in each drawer at each location ---
        print("[Exploration] Trying to place objects in drawers...")
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] execute_place: {obj} in {drawer} at {loc}")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        if done:
                            print(f"[Exploration] Task ended after execute_place {obj} in {drawer} at {loc}")
                            return
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # --- Step 6: Try to push each drawer at each location ---
        print("[Exploration] Trying to push drawers...")
        for drawer in drawers:
            for loc in locations:
                try:
                    print(f"[Exploration] execute_push: {drawer} at {loc}")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    if done:
                        print(f"[Exploration] Task ended after execute_push {drawer} at {loc}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # --- Step 7: Try to sweep each object at each location ---
        print("[Exploration] Trying to sweep objects...")
        for obj in objects:
            for loc in locations:
                try:
                    print(f"[Exploration] execute_sweep: {obj} at {loc}")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after execute_sweep {obj} at {loc}")
                        return
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # --- Step 8: Try to use gripper (no parameters) ---
        print("[Exploration] Trying to use gripper...")
        try:
            obs, reward, done = execute_gripper(env, task)
            if done:
                print(f"[Exploration] Task ended after execute_gripper")
                return
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # --- Step 9: Try to rotate (if available) ---
        if 'execute_rotate' in globals():
            print("[Exploration] Trying to rotate...")
            try:
                obs, reward, done = execute_rotate(env, task)
                if done:
                    print(f"[Exploration] Task ended after execute_rotate")
                    return
            except Exception as e:
                print(f"[Exploration] execute_rotate failed: {e}")

        print("[Exploration] Exploration phase completed. Check logs for failed actions and missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()