# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will systematically try each available skill and observe the environment for changes,
        # to infer which predicate is missing or required.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # For exploration, we will attempt to execute each skill with plausible arguments.
        # We will log the outcome and any errors to help identify missing predicates.

        # Get all objects, drawers, locations, switches from positions (if available)
        # For demonstration, we assume positions dict contains keys for 'objects', 'drawers', 'locations', 'switches'
        objects = positions.get('objects', [])
        drawers = positions.get('drawers', [])
        locations = positions.get('locations', [])
        switches = positions.get('switches', [])

        # Helper: Try to execute a skill and catch errors
        def try_skill(skill_fn, *args, **kwargs):
            try:
                print(f"[Exploration] Trying {skill_fn.__name__} with args {args} kwargs {kwargs}")
                result = skill_fn(env, task, *args, **kwargs)
                print(f"[Exploration] {skill_fn.__name__} succeeded.")
                return result
            except Exception as e:
                print(f"[Exploration] {skill_fn.__name__} failed: {e}")
                return None

        # 1. Try execute_go between locations (if at least two locations exist)
        if len(locations) >= 2:
            from_loc = locations[0]
            to_loc = locations[1]
            try:
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
                print(f"[Exploration] execute_go from {from_loc} to {to_loc} result: done={done}")
            except Exception as e:
                print(f"[Exploration] execute_go failed: {e}")

        # 2. Try execute_push_switch if a switch and location exist
        if switches and locations:
            switch = switches[0]
            loc = locations[0]
            try:
                obs, reward, done = execute_push_switch(env, task, switch, loc)
                print(f"[Exploration] execute_push_switch at {loc} result: done={done}")
            except Exception as e:
                print(f"[Exploration] execute_push_switch failed: {e}")

        # 3. Try execute_pick for each object at each location
        for obj in objects:
            for loc in locations:
                try:
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    print(f"[Exploration] execute_pick {obj} at {loc} result: done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

        # 4. Try execute_place for each object, drawer, and location
        for obj in objects:
            for drawer in drawers:
                for loc in locations:
                    try:
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                        print(f"[Exploration] execute_place {obj} in {drawer} at {loc} result: done={done}")
                    except Exception as e:
                        print(f"[Exploration] execute_place failed for {obj}, {drawer} at {loc}: {e}")

        # 5. Try execute_pull for each drawer, handle (object), and location
        for drawer in drawers:
            for handle in objects:
                for loc in locations:
                    try:
                        obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                        print(f"[Exploration] execute_pull {drawer} with {handle} at {loc} result: done={done}")
                    except Exception as e:
                        print(f"[Exploration] execute_pull failed for {drawer}, {handle} at {loc}: {e}")

        # 6. Try execute_push for each drawer and location
        for drawer in drawers:
            for loc in locations:
                try:
                    obs, reward, done = execute_push(env, task, drawer, loc)
                    print(f"[Exploration] execute_push {drawer} at {loc} result: done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")

        # 7. Try execute_sweep for each object and location
        for obj in objects:
            for loc in locations:
                try:
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                    print(f"[Exploration] execute_sweep {obj} at {loc} result: done={done}")
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")

        # 8. Try execute_gripper (no arguments)
        try:
            obs, reward, done = execute_gripper(env, task)
            print(f"[Exploration] execute_gripper result: done={done}")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Review above logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()