# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, e.g., (robot-at floor)
        # We will use available skills to explore and confirm the robot's location predicates.

        # For demonstration, we will attempt to use the available skills to move the robot and observe the effect.
        # Since we do not have the actual oracle plan, we focus on exploration to identify the missing predicate.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to use 'execute_go' to move the robot and check if the predicate (robot-at floor) is set
        # We assume 'floor' is a location in the environment

        # Attempt to get all locations from positions
        location_names = []
        for name, pos in positions.items():
            if 'floor' in name or 'room' in name or 'drawer' in name or 'location' in name:
                location_names.append(name)
        # Fallback: if no locations found, just use all keys
        if not location_names:
            location_names = list(positions.keys())

        # For exploration, try to move the robot to each location using execute_go
        # and observe if the predicate (robot-at <location>) is set
        for i, to_location in enumerate(location_names):
            try:
                # For the first move, assume the robot starts at the first location
                if i == 0:
                    from_location = to_location
                    continue
                # Use execute_go to move from previous location to current
                print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} using execute_go.")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_location,
                    to_location=to_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Exploration] Robot moved to {to_location}.")
                # After moving, check if the predicate (robot-at <to_location>) is set
                # (In a real system, you would check the state or observation here)
                from_location = to_location
                if done:
                    print("[Exploration] Task ended after move.")
                    break
            except Exception as e:
                print(f"[Exploration] Failed to move robot from {from_location} to {to_location}: {e}")
                continue

        # After exploration, print a message about the missing predicate
        print("[Exploration] Based on feedback and exploration, the missing predicate is likely (robot-at floor).")
        print("[Exploration] Please ensure that the initial state includes (robot-at floor) or the correct location predicate.")

        # === End of Exploration Phase ===

        # (If you had an oracle plan, you would execute it here using the available skills.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()