# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The feedback indicates that the planner times out, likely due to a missing predicate in the initial state.
        # We will perform an exploration phase to try to discover which predicate is missing by using available skills.

        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe which predicates are affected.

        # Since we do not have the actual object and location names, we will use placeholders and try to generalize.
        # You should replace 'object_1', 'drawer_1', 'location_1', etc. with actual names from your environment if available.

        # Example object and location keys (replace with actual keys from your environment)
        object_keys = [k for k in positions.keys() if 'object' in k or 'obj' in k]
        drawer_keys = [k for k in positions.keys() if 'drawer' in k]
        location_keys = [k for k in positions.keys() if 'location' in k or 'room' in k or 'area' in k]

        # Fallbacks if keys are not found
        if not object_keys:
            object_keys = ['object_1']
        if not drawer_keys:
            drawer_keys = ['drawer_1']
        if not location_keys:
            location_keys = ['location_1', 'location_2']

        # For demonstration, pick the first object, drawer, and two locations
        obj = object_keys[0]
        drawer = drawer_keys[0]
        loc_from = location_keys[0]
        loc_to = location_keys[1] if len(location_keys) > 1 else location_keys[0]

        # 1. Try to move the robot to a new location (explore execute_go)
        try:
            print(f"[Exploration] Trying execute_go from {loc_from} to {loc_to}")
            obs, reward, done = execute_go(env, task, loc_from, loc_to)
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick up an object (explore execute_pick)
        try:
            print(f"[Exploration] Trying execute_pick on {obj} at {loc_to}")
            obs, reward, done = execute_pick(env, task, obj, loc_to)
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to place the object in the drawer (explore execute_place)
        try:
            print(f"[Exploration] Trying execute_place {obj} in {drawer} at {loc_to}")
            obs, reward, done = execute_place(env, task, obj, drawer, loc_to)
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")

        # 4. Try to open the drawer (explore execute_pull)
        # For execute_pull, we need a handle object. We'll try with the same object as a placeholder.
        try:
            print(f"[Exploration] Trying execute_pull on {drawer} with handle {obj} at {loc_to}")
            obs, reward, done = execute_pull(env, task, drawer, obj, loc_to)
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # 5. Try to push (close) the drawer (explore execute_push)
        try:
            print(f"[Exploration] Trying execute_push on {drawer} at {loc_to}")
            obs, reward, done = execute_push(env, task, drawer, loc_to)
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to sweep the object (explore execute_sweep)
        try:
            print(f"[Exploration] Trying execute_sweep on {obj} at {loc_to}")
            obs, reward, done = execute_sweep(env, task, obj, loc_to)
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try to use the gripper (explore execute_gripper)
        try:
            print(f"[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # 8. Try to push the switch (explore execute_push_switch)
        # Note: The skill name in the list is 'execute_push_swtich -> push', but the correct function is likely 'execute_push_switch'
        # We'll try both spellings for robustness.
        try:
            print(f"[Exploration] Trying execute_push_switch on 'switch_1' at {loc_to}")
            obs, reward, done = execute_push_switch(env, task, 'switch_1', loc_to)
        except Exception as e:
            print(f"[Exploration] execute_push_switch failed: {e}")
        try:
            print(f"[Exploration] Trying execute_push_swtich on 'switch_1' at {loc_to}")
            obs, reward, done = execute_push_swtich(env, task, 'switch_1', loc_to)
        except Exception as e:
            print(f"[Exploration] execute_push_swtich failed: {e}")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase complete. Check logs and environment state for missing predicate clues.")

        # TODO: After exploration, analyze the environment state and logs to determine which predicate is missing.
        # This may require human inspection or further automated analysis.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
