# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We use the exploration domain knowledge to try to discover which predicate is missing.
        # This is done by attempting to execute exploration skills and observing the environment's response.

        # For demonstration, we attempt to use available skills to probe the environment.
        # We assume the robot and objects are named in the positions dictionary.
        # The actual names and types should be adapted to your environment.

        # Example: Try to identify all objects at all locations
        print("[Exploration] Attempting to identify objects at all locations...")
        for obj_name, obj_pos in positions.items():
            try:
                # Try to execute a pick to see if the object can be picked (weight/durability known?)
                print(f"[Exploration] Trying to pick {obj_name} at {obj_pos}...")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj_name}.")
                else:
                    print(f"[Exploration] Could not pick {obj_name}.")
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj_name}: {e}")

            try:
                # Try to execute a sweep (if available) to see if the object can be swept
                print(f"[Exploration] Trying to sweep {obj_name} at {obj_pos}...")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj_name}.")
                else:
                    print(f"[Exploration] Could not sweep {obj_name}.")
            except Exception as e:
                print(f"[Exploration] Exception during sweep of {obj_name}: {e}")

        # Try to move the robot to all locations to see if any location is problematic
        print("[Exploration] Attempting to move robot to all locations...")
        location_names = [name for name in positions if 'location' in name or 'room' in name]
        if len(location_names) >= 2:
            from_loc = location_names[0]
            to_loc = location_names[1]
            try:
                print(f"[Exploration] Trying to go from {from_loc} to {to_loc}...")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=positions[from_loc],
                    to_location=positions[to_loc],
                    max_steps=100,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}.")
                else:
                    print(f"[Exploration] Could not move from {from_loc} to {to_loc}.")
            except Exception as e:
                print(f"[Exploration] Exception during go from {from_loc} to {to_loc}: {e}")

        # Try to manipulate drawers if present
        drawer_names = [name for name in positions if 'drawer' in name]
        for drawer_name in drawer_names:
            try:
                print(f"[Exploration] Trying to push drawer {drawer_name}...")
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer_name=drawer_name,
                    location=positions[drawer_name],
                    max_steps=100,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully pushed drawer {drawer_name}.")
                else:
                    print(f"[Exploration] Could not push drawer {drawer_name}.")
            except Exception as e:
                print(f"[Exploration] Exception during push of drawer {drawer_name}: {e}")

        # Try to turn on the light if the room is dark
        switch_names = [name for name in positions if 'switch' in name]
        for switch_name in switch_names:
            try:
                print(f"[Exploration] Trying to push switch {switch_name}...")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_name=switch_name,
                    location=positions[switch_name],
                    max_steps=100,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully pushed switch {switch_name}.")
                else:
                    print(f"[Exploration] Could not push switch {switch_name}.")
            except Exception as e:
                print(f"[Exploration] Exception during push of switch {switch_name}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === End of Exploration Phase ===

        # TODO: After exploration, use the discovered information to construct and execute the oracle plan.
        # For now, exploration is the main focus as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()