# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and try to discover the missing predicate.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe the effects.

        # Get all objects and locations from positions
        object_names = [name for name in positions if positions[name].get('type', '') == 'object']
        drawer_names = [name for name in positions if positions[name].get('type', '') == 'drawer']
        location_names = [name for name in positions if positions[name].get('type', '') == 'location']
        switch_names = [name for name in positions if positions[name].get('type', '') == 'switch']

        # Fallback: If types are not provided, infer by name
        if not object_names:
            object_names = [name for name in positions if 'object' in name or 'obj' in name]
        if not drawer_names:
            drawer_names = [name for name in positions if 'drawer' in name]
        if not location_names:
            location_names = [name for name in positions if 'room' in name or 'loc' in name]
        if not switch_names:
            switch_names = [name for name in positions if 'switch' in name]

        # Try to get robot's initial location
        robot_location = None
        for loc in location_names:
            if positions[loc].get('robot_at', False):
                robot_location = loc
                break
        if robot_location is None and location_names:
            robot_location = location_names[0]

        # 1. Try to turn on the light if the room is dark (execute_push_switch)
        #    This is required for most actions in the domain.
        if switch_names and robot_location:
            try:
                print("[Exploration] Trying to turn on the light using execute_push_switch.")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    switch_names[0],
                    robot_location
                )
            except Exception as e:
                print("[Exploration] execute_push_switch failed:", e)

        # 2. Try to pick up each object (execute_pick)
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to pick up object {obj} using execute_pick.")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully picked up {obj}.")
                    break
            except Exception as e:
                print(f"[Exploration] execute_pick failed for {obj}:", e)

        # 3. Try to open each drawer (execute_pull)
        for drawer in drawer_names:
            # Find a handle object for the drawer
            handle_obj = None
            for obj in object_names:
                if positions[obj].get('handle_of', '') == drawer:
                    handle_obj = obj
                    break
            if handle_obj:
                try:
                    print(f"[Exploration] Trying to open drawer {drawer} with handle {handle_obj} using execute_pull.")
                    obs, reward, done = execute_pull(
                        env,
                        task,
                        drawer,
                        handle_obj,
                        robot_location
                    )
                    if done:
                        print(f"[Exploration] Successfully opened drawer {drawer}.")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_pull failed for {drawer}:", e)

        # 4. Try to place an object in a drawer (execute_place)
        for obj in object_names:
            for drawer in drawer_names:
                try:
                    print(f"[Exploration] Trying to place {obj} in {drawer} using execute_place.")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        obj,
                        drawer,
                        robot_location
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in {drawer}.")
                        break
                except Exception as e:
                    print(f"[Exploration] execute_place failed for {obj} in {drawer}:", e)

        # 5. Try to move to another location (execute_go)
        for to_loc in location_names:
            if to_loc != robot_location:
                try:
                    print(f"[Exploration] Trying to move from {robot_location} to {to_loc} using execute_go.")
                    obs, reward, done = execute_go(
                        env,
                        task,
                        robot_location,
                        to_loc
                    )
                    if done:
                        print(f"[Exploration] Successfully moved to {to_loc}.")
                        robot_location = to_loc
                        break
                except Exception as e:
                    print(f"[Exploration] execute_go failed from {robot_location} to {to_loc}:", e)

        # 6. Try to sweep the floor (execute_sweep)
        for obj in object_names:
            try:
                print(f"[Exploration] Trying to sweep object {obj} using execute_sweep.")
                obs, reward, done = execute_sweep(
                    env,
                    task,
                    obj,
                    robot_location
                )
                if done:
                    print(f"[Exploration] Successfully swept {obj}.")
                    break
            except Exception as e:
                print(f"[Exploration] execute_sweep failed for {obj}:", e)

        # 7. Try to use the gripper (execute_gripper)
        try:
            print("[Exploration] Trying to use gripper using execute_gripper.")
            obs, reward, done = execute_gripper(
                env,
                task
            )
            if done:
                print("[Exploration] Successfully used gripper.")
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()