# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available skills to explore the environment and infer missing predicates.
        # For this, we try to execute each available skill in a safe, exploratory way and observe the results.

        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # For demonstration, we will attempt to use each skill on available objects/locations
        # and catch exceptions to infer which predicates might be missing.

        # Get lists of objects, drawers, locations, switches from positions (if available)
        objects = []
        drawers = []
        locations = []
        switches = []
        for name, pos in positions.items():
            if 'drawer' in name:
                drawers.append(name)
            elif 'switch' in name:
                switches.append(name)
            elif 'loc' in name or 'room' in name:
                locations.append(name)
            else:
                objects.append(name)
        # Fallback: if no locations found, use all unique positions as locations
        if not locations:
            locations = list(set([v for v in positions.values()]))

        # Try to execute each skill in a safe way
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # Try to pick the first object at the first location
                    if objects and locations:
                        obs, reward, done = execute_pick(
                            env, task, obj=objects[0], p=locations[0]
                        )
                        print(f"[Exploration] execute_pick succeeded on {objects[0]} at {locations[0]}")
                elif skill == 'execute_place':
                    # Try to place the first object into the first drawer at the first location
                    if objects and drawers and locations:
                        obs, reward, done = execute_place(
                            env, task, o=objects[0], d=drawers[0], p=locations[0]
                        )
                        print(f"[Exploration] execute_place succeeded for {objects[0]} in {drawers[0]} at {locations[0]}")
                elif skill == 'execute_push':
                    # Try to push the first drawer at the first location
                    if drawers and locations:
                        obs, reward, done = execute_push(
                            env, task, d=drawers[0], p=locations[0]
                        )
                        print(f"[Exploration] execute_push succeeded for {drawers[0]} at {locations[0]}")
                elif skill == 'execute_pull':
                    # Try to pull the first drawer handle at the first location
                    if drawers and objects and locations:
                        obs, reward, done = execute_pull(
                            env, task, d=drawers[0], h=objects[0], p=locations[0]
                        )
                        print(f"[Exploration] execute_pull succeeded for {drawers[0]} with handle {objects[0]} at {locations[0]}")
                elif skill == 'execute_sweep':
                    # Try to sweep the first object at the first location
                    if objects and locations:
                        obs, reward, done = execute_sweep(
                            env, task, o=objects[0], p=locations[0]
                        )
                        print(f"[Exploration] execute_sweep succeeded for {objects[0]} at {locations[0]}")
                elif skill == 'execute_rotate':
                    # If rotate is implemented, try to call it (parameters may vary)
                    if hasattr(env, 'execute_rotate'):
                        obs, reward, done = execute_rotate(env, task)
                        print(f"[Exploration] execute_rotate succeeded")
                elif skill == 'execute_go':
                    # Try to move from the first location to the second location
                    if len(locations) >= 2:
                        obs, reward, done = execute_go(
                            env, task, from_loc=locations[0], to_loc=locations[1]
                        )
                        print(f"[Exploration] execute_go succeeded from {locations[0]} to {locations[1]}")
                elif skill == 'execute_gripper':
                    # Try to operate the gripper
                    obs, reward, done = execute_gripper(env, task)
                    print(f"[Exploration] execute_gripper succeeded")
                elif skill.startswith('execute_push_swtich'):
                    # Try to push the first switch at the first location
                    if switches and locations:
                        obs, reward, done = execute_push_switch(
                            env, task, s=switches[0], p=locations[0]
                        )
                        print(f"[Exploration] execute_push_switch succeeded for {switches[0]} at {locations[0]}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed: {e}")

        # After exploration, print a message to indicate that missing predicates may have been identified
        print("[Exploration] Exploration phase complete. Check logs for failed skills and missing predicates.")

        # === (Optional) Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this template, we focus on the exploration phase as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
