# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor)
        # The missing predicate is likely related to the robot's location, specifically (robot-at floor).
        # We will use available skills to explore and confirm the robot's location.

        # The available skills are:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will use 'execute_go' to move the robot and check if the predicate (robot-at floor) is satisfied.

        # For demonstration, let's assume 'floor' is a location in the environment.
        # We'll try to move the robot to 'floor' using 'execute_go' and observe the result.

        # Get all locations from positions (assuming keys with type 'location')
        # If positions dict contains location names, extract them
        location_names = []
        for name, pos in positions.items():
            if 'location' in name or 'floor' in name or 'room' in name:
                location_names.append(name)
        if 'floor' not in location_names:
            location_names.append('floor')  # Ensure 'floor' is included for exploration

        # Try to get the robot's current location
        robot_location = None
        for name in location_names:
            if 'robot' in name or 'robby' in name:
                robot_location = name
                break
        if robot_location is None:
            # Fallback: try to get from observation or assume starting at first location
            if len(location_names) > 0:
                robot_location = location_names[0]
            else:
                robot_location = 'floor'

        # Exploration: Try to move the robot to 'floor' using execute_go
        # This will help us confirm if (robot-at floor) is a valid predicate/state

        print("[Exploration] Attempting to move robot to 'floor' to check (robot-at floor) predicate.")
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location='floor',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] execute_go to 'floor' executed successfully.")
        except Exception as e:
            print(f"[Exploration] Failed to execute_go to 'floor': {e}")

        # After moving, check if the robot is at 'floor'
        # This can be done by checking the observation or positions if available
        try:
            updated_positions = get_object_positions()
            robot_at_floor = False
            for name, pos in updated_positions.items():
                if 'robot' in name or 'robby' in name:
                    # If the robot's position matches 'floor' position (if available)
                    if 'floor' in updated_positions:
                        if np.allclose(pos, updated_positions['floor'], atol=0.05):
                            robot_at_floor = True
                            break
            if robot_at_floor:
                print("[Exploration] Robot is at 'floor'. Predicate (robot-at floor) is TRUE.")
            else:
                print("[Exploration] Robot is NOT at 'floor'. Predicate (robot-at floor) is FALSE or missing.")
        except Exception as e:
            print(f"[Exploration] Could not verify robot-at floor predicate: {e}")

        # === End of Exploration Phase ===

        # TODO: Continue with further task plan steps as needed, using only predefined skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
