# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to execute available skills in a diagnostic sequence to discover which predicate is missing.
        # This phase is for exploration and debugging.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill with plausible arguments to see if any precondition fails
        # and log the result for debugging the missing predicate.

        # For demonstration, we will attempt to execute each skill with dummy or inferred arguments.
        # In a real scenario, you would use actual object/location names from the environment.

        # Example: Try to pick up an object from the floor
        try:
            # Find an object on the floor and a location
            object_on_floor = None
            location = None
            for obj_name, pos in positions.items():
                if 'floor' in obj_name or 'object' in obj_name:
                    object_on_floor = obj_name
                    break
            for loc_name in positions.keys():
                if 'room' in loc_name or 'location' in loc_name:
                    location = loc_name
                    break
            if object_on_floor and location:
                print(f"[Exploration] Trying execute_pick on {object_on_floor} at {location}")
                obs, reward, done = execute_pick(env, task, object_on_floor, location)
            else:
                print("[Exploration] Could not find suitable object/location for execute_pick")
        except Exception as e:
            print(f"[Exploration] execute_pick failed: {e}")

        # Try to open a drawer (execute_pull)
        try:
            drawer = None
            handle = None
            for obj_name in positions.keys():
                if 'drawer' in obj_name:
                    drawer = obj_name
                if 'handle' in obj_name:
                    handle = obj_name
            if drawer and handle and location:
                print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle} at {location}")
                obs, reward, done = execute_pull(env, task, drawer, handle, location)
            else:
                print("[Exploration] Could not find suitable drawer/handle/location for execute_pull")
        except Exception as e:
            print(f"[Exploration] execute_pull failed: {e}")

        # Try to move (execute_go) between two locations
        try:
            locations = [k for k in positions.keys() if 'room' in k or 'location' in k]
            if len(locations) >= 2:
                from_loc, to_loc = locations[0], locations[1]
                print(f"[Exploration] Trying execute_go from {from_loc} to {to_loc}")
                obs, reward, done = execute_go(env, task, from_loc, to_loc)
            else:
                print("[Exploration] Not enough locations for execute_go")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # Try to push a drawer closed (execute_push)
        try:
            if drawer and location:
                print(f"[Exploration] Trying execute_push on {drawer} at {location}")
                obs, reward, done = execute_push(env, task, drawer, location)
            else:
                print("[Exploration] Could not find suitable drawer/location for execute_push")
        except Exception as e:
            print(f"[Exploration] execute_push failed: {e}")

        # Try to place an object in a drawer (execute_place)
        try:
            if object_on_floor and drawer and location:
                print(f"[Exploration] Trying execute_place {object_on_floor} in {drawer} at {location}")
                obs, reward, done = execute_place(env, task, object_on_floor, drawer, location)
            else:
                print("[Exploration] Could not find suitable object/drawer/location for execute_place")
        except Exception as e:
            print(f"[Exploration] execute_place failed: {e}")

        # Try to sweep (execute_sweep)
        try:
            if object_on_floor and location:
                print(f"[Exploration] Trying execute_sweep on {object_on_floor} at {location}")
                obs, reward, done = execute_sweep(env, task, object_on_floor, location)
            else:
                print("[Exploration] Could not find suitable object/location for execute_sweep")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        # Try to rotate (execute_rotate) if available
        try:
            # Not enough info for arguments, so just call with no arguments if possible
            print(f"[Exploration] Trying execute_rotate (no arguments)")
            obs, reward, done = execute_rotate(env, task)
        except Exception as e:
            print(f"[Exploration] execute_rotate failed: {e}")

        # Try to use gripper (execute_gripper)
        try:
            print(f"[Exploration] Trying execute_gripper (no arguments)")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        print("[Exploration] Exploration phase complete. Check above for missing predicate clues.")

        # === End of Exploration Phase ===

        # === (Optional) Main Task Plan Execution ===
        # After identifying the missing predicate, you would implement the actual oracle plan here
        # using only the available skills and correct arguments.
        # For now, this code focuses on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()