# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at floor) is missing
        # We need to explore to determine the correct predicate for the robot's location

        # Try to use available skills to explore the environment and check for missing predicates
        # Since we cannot define new skills, we use the available ones to move and observe

        # Let's print out all available positions for debugging
        print("[Exploration] Object positions:", positions)

        # Try to find the robot's current location
        robot_location = None
        for name, pos in positions.items():
            if "robot" in name or "base" in name:
                robot_location = name
                break

        if robot_location is None:
            # Fallback: try to infer from observation
            robot_location = "floor"  # As per feedback

        print(f"[Exploration] Robot initial location assumed as: {robot_location}")

        # Try to execute a go/move action to another location if possible
        # Find another location to move to
        possible_locations = [name for name in positions.keys() if name != robot_location]
        if possible_locations:
            target_location = possible_locations[0]
        else:
            target_location = robot_location  # fallback

        # Use execute_go if available to move the robot and see if the predicate changes
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {target_location}")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=target_location,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {target_location}")
        except Exception as e:
            print(f"[Exploration] execute_go failed: {e}")

        # After moving, check if the predicate (robot-at <location>) is now true
        # This is a placeholder for actual predicate checking, which would be handled by the planner or environment
        print(f"[Exploration] After move, check if (robot-at {target_location}) holds in the environment.")

        # === End of Exploration Phase ===

        # === Main Task Plan Execution ===
        # Here you would execute the oracle plan using only the available skills.
        # For demonstration, let's try to pick and place an object if possible.

        # Find an object on the floor to pick
        object_to_pick = None
        for name in positions.keys():
            if "object" in name or "ball" in name or "cube" in name:
                object_to_pick = name
                break

        if object_to_pick:
            try:
                print(f"[Task] Attempting to pick up {object_to_pick} at {target_location}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_to_pick,
                    location=target_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Picked up {object_to_pick}")
            except Exception as e:
                print(f"[Task] execute_pick failed: {e}")

        # If there is a drawer, try to place the object in it
        drawer_name = None
        for name in positions.keys():
            if "drawer" in name:
                drawer_name = name
                break

        if object_to_pick and drawer_name:
            try:
                print(f"[Task] Attempting to place {object_to_pick} in {drawer_name} at {target_location}")
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj_name=object_to_pick,
                    drawer_name=drawer_name,
                    location=target_location,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Placed {object_to_pick} in {drawer_name}")
            except Exception as e:
                print(f"[Task] execute_place failed: {e}")

        # You can continue with more plan steps as needed, using only the available skills

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()