# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a planning failure, likely due to a missing predicate in the domain/problem.
        # We perform an exploration phase using available skills to discover which predicate is missing.
        # The exploration domain provides actions like execute_go_identify, execute_go_temperature, etc.
        # However, in this environment, we only use the available skills.

        # For demonstration, we attempt to use the available skills to interact with objects and locations,
        # and log any failures or unexpected results to infer missing predicates.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # Example: Try to pick up all objects on the floor at their locations
        # and attempt to place them in a drawer, logging any failures.
        # This helps to identify if a predicate like (on-floor ?o) or (drawer-open ?d) is missing.

        # Get all objects and locations from positions
        object_names = [name for name in positions if 'object' in name]
        location_names = [name for name in positions if 'location' in name or 'room' in name or 'drawer' in name]
        drawer_names = [name for name in positions if 'drawer' in name]

        # For exploration, try to pick each object and place it in a drawer
        for obj in object_names:
            try:
                obj_pos = positions[obj]
                # Try to pick the object at its location
                print(f"[Exploration] Attempting to pick {obj} at {obj_pos}")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    target_obj=obj,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully picked {obj}")
                else:
                    print(f"[Exploration] Failed to pick {obj} - possible missing predicate or precondition")
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj}: {e}")

            # Try to place the object in each drawer
            for drawer in drawer_names:
                try:
                    drawer_pos = positions[drawer]
                    print(f"[Exploration] Attempting to place {obj} in {drawer} at {drawer_pos}")
                    obs, reward, done = execute_place(
                        env,
                        task,
                        target_obj=obj,
                        target_drawer=drawer,
                        target_pos=drawer_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Successfully placed {obj} in {drawer}")
                    else:
                        print(f"[Exploration] Failed to place {obj} in {drawer} - possible missing predicate or precondition")
                except Exception as e:
                    print(f"[Exploration] Exception during place of {obj} in {drawer}: {e}")

        # Try to open and close each drawer to check for lock or state predicates
        for drawer in drawer_names:
            try:
                drawer_pos = positions[drawer]
                # Try to pull (open) the drawer
                print(f"[Exploration] Attempting to pull (open) {drawer} at {drawer_pos}")
                obs, reward, done = execute_pull(
                    env,
                    task,
                    target_drawer=drawer,
                    target_pos=drawer_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully pulled (opened) {drawer}")
                else:
                    print(f"[Exploration] Failed to pull (open) {drawer} - possible missing predicate or precondition")
            except Exception as e:
                print(f"[Exploration] Exception during pull (open) of {drawer}: {e}")

            try:
                # Try to push (close) the drawer
                print(f"[Exploration] Attempting to push (close) {drawer} at {drawer_pos}")
                obs, reward, done = execute_push(
                    env,
                    task,
                    target_drawer=drawer,
                    target_pos=drawer_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully pushed (closed) {drawer}")
                else:
                    print(f"[Exploration] Failed to push (close) {drawer} - possible missing predicate or precondition")
            except Exception as e:
                print(f"[Exploration] Exception during push (close) of {drawer}: {e}")

        # Try to toggle the light switch if available
        switch_names = [name for name in positions if 'switch' in name]
        for switch in switch_names:
            try:
                switch_pos = positions[switch]
                print(f"[Exploration] Attempting to push switch {switch} at {switch_pos}")
                obs, reward, done = execute_push_switch(
                    env,
                    task,
                    target_switch=switch,
                    target_pos=switch_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Successfully pushed switch {switch}")
                else:
                    print(f"[Exploration] Failed to push switch {switch} - possible missing predicate or precondition")
            except Exception as e:
                print(f"[Exploration] Exception during push switch {switch}: {e}")

        # Try to move between locations
        for from_loc in location_names:
            for to_loc in location_names:
                if from_loc != to_loc:
                    try:
                        print(f"[Exploration] Attempting to go from {from_loc} to {to_loc}")
                        obs, reward, done = execute_go(
                            env,
                            task,
                            from_location=from_loc,
                            to_location=to_loc,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Successfully moved from {from_loc} to {to_loc}")
                        else:
                            print(f"[Exploration] Failed to move from {from_loc} to {to_loc} - possible missing predicate or precondition")
                    except Exception as e:
                        print(f"[Exploration] Exception during go from {from_loc} to {to_loc}: {e}")

        # After exploration, the log will indicate which actions failed and may hint at missing predicates.
        # The user can then update the domain/problem PDDL accordingly.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
