# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Find Missing Predicate (room-bright) ===
        # The feedback indicates that (room-bright) is missing or not achieved.
        # According to the domain, (room-bright) is achieved by execute_push_switch
        # when the room is dark, the robot is free, and at the correct location.

        # We will attempt to execute the skill 'execute_push_switch' if available.
        # Since 'execute_push_switch' is not in the available skill list, but 'execute_gripper' and 'execute_sweep' are,
        # and 'execute_go' is available for moving, we will try to explore the environment using available skills.

        # The goal is to discover if the room is dark and, if so, to try to make it bright using available skills.

        # Step 1: Check if the room is dark (simulate exploration)
        # Since we cannot directly check predicates, we attempt actions that require (room-bright)
        # and handle failure gracefully.

        # Step 2: Try to execute a skill that requires (room-bright) and see if it fails.
        # If it fails, we infer that (room-bright) is missing.

        # For demonstration, we attempt to pick an object (requires room-bright).
        # If it fails, we know room-bright is missing and needs to be achieved.

        # Find an object on the floor and its location
        object_on_floor = None
        object_location = None
        for obj_name, pos in positions.items():
            if 'floor' in obj_name or 'object' in obj_name:
                object_on_floor = obj_name
                object_location = pos
                break

        # If no object found, just pick any available object
        if object_on_floor is None and len(positions) > 0:
            object_on_floor = list(positions.keys())[0]
            object_location = positions[object_on_floor]

        # Try to execute_pick (requires room-bright)
        pick_success = False
        if object_on_floor is not None:
            try:
                print(f"[Exploration] Attempting to pick {object_on_floor} at {object_location} (should require room-bright).")
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj_name=object_on_floor,
                    location=object_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                pick_success = True
                print("[Exploration] Pick succeeded. (room-bright) is present.")
            except Exception as e:
                print(f"[Exploration] Pick failed: {e}")
                print("[Exploration] (room-bright) is likely missing. Need to achieve (room-bright).")

        # Step 3: Try to achieve (room-bright) using available skills
        # Since 'execute_push_switch' is not in the available skills, we cannot directly turn on the light.
        # Instead, we can try to use other skills to explore or manipulate the environment.

        # Try to use execute_sweep or execute_gripper as exploration actions
        try:
            print("[Exploration] Attempting to use execute_sweep to explore environment.")
            obs, reward, done = execute_sweep(
                env,
                task,
                obj_name=object_on_floor,
                location=object_location,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print("[Exploration] execute_sweep executed.")
        except Exception as e:
            print(f"[Exploration] execute_sweep failed: {e}")

        try:
            print("[Exploration] Attempting to use execute_gripper to explore environment.")
            obs, reward, done = execute_gripper(
                env,
                task,
                max_steps=50,
                timeout=5.0
            )
            print("[Exploration] execute_gripper executed.")
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # Step 4: Try to move to another location using execute_go (requires room-bright)
        # Find two locations to move between
        locations = [name for name in positions if 'location' in name or 'room' in name]
        if len(locations) >= 2:
            from_location = locations[0]
            to_location = locations[1]
            try:
                print(f"[Exploration] Attempting to move from {from_location} to {to_location} (should require room-bright).")
                obs, reward, done = execute_go(
                    env,
                    task,
                    from_location=from_location,
                    to_location=to_location,
                    max_steps=100,
                    timeout=10.0
                )
                print("[Exploration] Move succeeded. (room-bright) is present.")
            except Exception as e:
                print(f"[Exploration] Move failed: {e}")
                print("[Exploration] (room-bright) is likely missing for movement as well.")

        # Step 5: Summarize exploration
        print("[Exploration] Exploration phase complete. If actions requiring (room-bright) failed, the missing predicate is (room-bright).")

        # === End of Exploration ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()