# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to execute available exploration skills to discover which predicate is missing.
        # The exploration domain provides skills like execute_go_identify, execute_go_temperature, etc.
        # However, we are restricted to only the provided skills in skill_code.
        # Thus, we will attempt to use the available skills to interact with the environment and observe failures.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # For exploration, we will try to execute each skill with plausible arguments
        # and catch any exceptions or errors to infer missing predicates.

        # Get all objects, drawers, locations, switches from positions (if available)
        # For demonstration, we assume positions dict contains keys for objects, drawers, locations, switches
        objects = [k for k in positions if 'object' in k]
        drawers = [k for k in positions if 'drawer' in k]
        locations = [k for k in positions if 'location' in k or 'room' in k]
        switches = [k for k in positions if 'switch' in k]

        # Fallback: If not found, use all keys as possible locations
        if not locations:
            locations = list(positions.keys())

        # Try to execute each skill and observe the result
        for skill in available_skills:
            try:
                print(f"[Exploration] Attempting skill: {skill}")
                if skill == 'execute_pick':
                    # Try picking each object at each location
                    for obj in objects:
                        for loc in locations:
                            try:
                                obs, reward, done = execute_pick(
                                    env, task, obj, loc
                                )
                                print(f"Executed execute_pick({obj}, {loc})")
                            except Exception as e:
                                print(f"execute_pick({obj}, {loc}) failed: {e}")
                elif skill == 'execute_place':
                    # Try placing each object into each drawer at each location
                    for obj in objects:
                        for drawer in drawers:
                            for loc in locations:
                                try:
                                    obs, reward, done = execute_place(
                                        env, task, obj, drawer, loc
                                    )
                                    print(f"Executed execute_place({obj}, {drawer}, {loc})")
                                except Exception as e:
                                    print(f"execute_place({obj}, {drawer}, {loc}) failed: {e}")
                elif skill == 'execute_push':
                    # Try pushing each drawer at each location
                    for drawer in drawers:
                        for loc in locations:
                            try:
                                obs, reward, done = execute_push(
                                    env, task, drawer, loc
                                )
                                print(f"Executed execute_push({drawer}, {loc})")
                            except Exception as e:
                                print(f"execute_push({drawer}, {loc}) failed: {e}")
                elif skill == 'execute_pull':
                    # Try pulling each drawer handle at each location
                    for drawer in drawers:
                        for obj in objects:
                            for loc in locations:
                                try:
                                    obs, reward, done = execute_pull(
                                        env, task, drawer, obj, loc
                                    )
                                    print(f"Executed execute_pull({drawer}, {obj}, {loc})")
                                except Exception as e:
                                    print(f"execute_pull({drawer}, {obj}, {loc}) failed: {e}")
                elif skill == 'execute_sweep':
                    # Try sweeping each object at each location
                    for obj in objects:
                        for loc in locations:
                            try:
                                obs, reward, done = execute_sweep(
                                    env, task, obj, loc
                                )
                                print(f"Executed execute_sweep({obj}, {loc})")
                            except Exception as e:
                                print(f"execute_sweep({obj}, {loc}) failed: {e}")
                elif skill == 'execute_rotate':
                    # If rotate skill takes no arguments
                    try:
                        obs, reward, done = execute_rotate(env, task)
                        print("Executed execute_rotate()")
                    except Exception as e:
                        print(f"execute_rotate() failed: {e}")
                elif skill == 'execute_go':
                    # Try moving between all pairs of locations
                    for from_loc in locations:
                        for to_loc in locations:
                            if from_loc != to_loc:
                                try:
                                    obs, reward, done = execute_go(
                                        env, task, from_loc, to_loc
                                    )
                                    print(f"Executed execute_go({from_loc}, {to_loc})")
                                except Exception as e:
                                    print(f"execute_go({from_loc}, {to_loc}) failed: {e}")
                elif skill == 'execute_gripper':
                    # If gripper skill takes no arguments
                    try:
                        obs, reward, done = execute_gripper(env, task)
                        print("Executed execute_gripper()")
                    except Exception as e:
                        print(f"execute_gripper() failed: {e}")
                elif skill.startswith('execute_push_swtich'):
                    # Try pushing each switch at each location
                    for switch in switches:
                        for loc in locations:
                            try:
                                obs, reward, done = execute_push_switch(
                                    env, task, switch, loc
                                )
                                print(f"Executed execute_push_switch({switch}, {loc})")
                            except Exception as e:
                                print(f"execute_push_switch({switch}, {loc}) failed: {e}")
            except Exception as e:
                print(f"[Exploration] Skill {skill} failed with error: {e}")

        print("[Exploration] Exploration phase complete. Check above for missing predicate clues.")

        # === Main Task Plan Execution ===
        # After exploration, you would normally execute the oracle plan here.
        # For this skeleton, you would insert the plan steps using the available skills,
        # using the correct arguments as determined by your plan and environment state.

        # Example (commented out, to be replaced with actual plan):
        # obs, reward, done = execute_push_switch(env, task, 'switch1', 'room1')
        # obs, reward, done = execute_pick(env, task, 'object1', 'room1')
        # obs, reward, done = execute_go(env, task, 'room1', 'room2')
        # obs, reward, done = execute_place(env, task, 'object1', 'drawer1', 'room2')
        # ... (continue plan)

        # TODO: Insert actual plan execution code here after exploration.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
