# run_skeleton_task.py (Completed for Exploration of Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (robot-at drawer-area)
        # The plan failed due to a missing predicate (robot-at drawer-area).
        # We need to explore the environment to determine if the robot can be at 'drawer-area'
        # and what predicate is required for further actions.

        # For this, we will attempt to move the robot to the 'drawer-area' using available skills.
        # We assume 'drawer-area' is a key in positions, or fallback to a default location.

        # Find a plausible starting location and the 'drawer-area'
        location_keys = [k for k in positions.keys() if 'area' in k or 'room' in k or 'location' in k]
        drawer_area_key = None
        for k in positions.keys():
            if 'drawer' in k and 'area' in k:
                drawer_area_key = k
                break
        if drawer_area_key is None:
            # Fallback: try 'drawer_area' or similar
            for k in positions.keys():
                if k.replace('-', '_') == 'drawer_area':
                    drawer_area_key = k
                    break
        if drawer_area_key is None:
            print("[Exploration] Could not find 'drawer-area' in object positions. Exploration aborted.")
            return

        # Find a plausible starting location (not the drawer-area)
        start_location_key = None
        for k in location_keys:
            if k != drawer_area_key:
                start_location_key = k
                break
        if start_location_key is None:
            # Fallback: pick any location that is not the drawer-area
            for k in positions.keys():
                if k != drawer_area_key:
                    start_location_key = k
                    break
        if start_location_key is None:
            print("[Exploration] Could not find a valid starting location. Exploration aborted.")
            return

        print(f"[Exploration] Attempting to move robot from '{start_location_key}' to '{drawer_area_key}'.")

        # Try to move the robot to the drawer-area using the available skill
        # The skill is 'execute_go' (from domain and available skills)
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=start_location_key,
                to_location=drawer_area_key,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            print(f"[Exploration] Called execute_go from '{start_location_key}' to '{drawer_area_key}'.")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")
            return

        # After moving, check if the robot is at the drawer-area
        # This is to confirm the predicate (robot-at drawer-area) is now true in the environment
        # (Assume get_object_positions() or task.get_observation() can be used to check)
        try:
            obs = task.get_observation()
            # Check if robot is at drawer-area (predicate)
            # This is domain-specific; we assume the observation contains this info
            robot_at_drawer_area = False
            if hasattr(obs, 'robot_location'):
                if obs.robot_location == drawer_area_key:
                    robot_at_drawer_area = True
            elif isinstance(obs, dict):
                # Try common keys
                if 'robot_location' in obs and obs['robot_location'] == drawer_area_key:
                    robot_at_drawer_area = True
                elif 'robot-at' in obs and obs['robot-at'] == drawer_area_key:
                    robot_at_drawer_area = True
            if robot_at_drawer_area:
                print(f"[Exploration] Predicate (robot-at {drawer_area_key}) is TRUE after execute_go.")
            else:
                print(f"[Exploration] Predicate (robot-at {drawer_area_key}) is still FALSE after execute_go.")
        except Exception as e:
            print(f"[Exploration] Exception during observation check: {e}")

        # --- End of Exploration Phase ---

        # (Optional) Continue with further plan steps if needed
        # For this exploration, we stop after confirming the missing predicate

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()