# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The plan failed because the predicate (robot-at drawer-area) was missing or not properly set.
        # We need to explore and ensure the robot can reach and recognize the 'drawer-area' location.

        # Try to move the robot to 'drawer-area' using available skills.
        # The available skills include 'execute_go', which moves the robot between locations.
        # We'll attempt to move the robot to 'drawer-area' and check if the predicate is set.

        # For demonstration, let's assume the robot starts at 'init-area' and needs to go to 'drawer-area'.
        # The actual names of locations and objects should be replaced with those from your environment.

        # Example location names (replace with actual ones from your environment if different)
        start_location = None
        drawer_area = None

        # Try to infer location names from positions dictionary
        for key in positions.keys():
            if 'drawer' in key and 'area' in key:
                drawer_area = key
            if 'init' in key or 'start' in key:
                start_location = key

        # Fallbacks if not found
        if drawer_area is None:
            # Try a generic fallback
            drawer_area = 'drawer-area'
        if start_location is None:
            # Try a generic fallback
            start_location = 'init-area'

        print(f"[Exploration] Attempting to move robot from {start_location} to {drawer_area} to check predicate.")

        try:
            # Use the execute_go skill to move the robot
            obs, reward, done = execute_go(
                env,
                task,
                from_location=start_location,
                to_location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] Robot moved to {drawer_area}.")
        except Exception as e:
            print(f"[Exploration] Failed to move robot to {drawer_area}: {e}")

        # After moving, check if the predicate (robot-at drawer-area) is now true.
        # This would typically be checked via the environment's state or observation.
        # For demonstration, print the current robot location if available.
        try:
            current_positions = get_object_positions()
            robot_pos = current_positions.get('robot', None)
            print(f"[Exploration] Robot current position: {robot_pos}")
        except Exception as e:
            print(f"[Exploration] Could not retrieve robot position: {e}")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the main plan using only predefined skills.
        # Example: pick an object from the floor and place it in the drawer.

        # Identify an object on the floor and the drawer
        object_on_floor = None
        drawer_name = None
        for key in positions.keys():
            if 'object' in key or 'item' in key:
                object_on_floor = key
            if 'drawer' in key and 'area' not in key:
                drawer_name = key

        if object_on_floor is None:
            object_on_floor = 'object_1'  # fallback
        if drawer_name is None:
            drawer_name = 'drawer_1'  # fallback

        # 1. Ensure the room is bright (if not, push the switch)
        try:
            # Try to execute the push switch skill if available
            obs, reward, done = execute_push_switch(
                env,
                task,
                switch_name='switch_1',  # Replace with actual switch name if needed
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Task] Room light switched on.")
        except Exception as e:
            print(f"[Task] Could not switch on the light (may already be on): {e}")

        # 2. Pick the object from the floor
        try:
            obs, reward, done = execute_pick(
                env,
                task,
                object_name=object_on_floor,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Picked up {object_on_floor}.")
        except Exception as e:
            print(f"[Task] Failed to pick {object_on_floor}: {e}")

        # 3. Open the drawer (if needed)
        try:
            # Assume handle is named 'handle_1' and is at the same location
            obs, reward, done = execute_pull(
                env,
                task,
                drawer_name=drawer_name,
                handle_name='handle_1',
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Drawer {drawer_name} opened.")
        except Exception as e:
            print(f"[Task] Failed to open drawer {drawer_name}: {e}")

        # 4. Place the object in the drawer
        try:
            obs, reward, done = execute_place(
                env,
                task,
                object_name=object_on_floor,
                drawer_name=drawer_name,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Placed {object_on_floor} in {drawer_name}.")
        except Exception as e:
            print(f"[Task] Failed to place {object_on_floor} in {drawer_name}: {e}")

        # 5. Close the drawer
        try:
            obs, reward, done = execute_push(
                env,
                task,
                drawer_name=drawer_name,
                location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Task] Drawer {drawer_name} closed.")
        except Exception as e:
            print(f"[Task] Failed to close drawer {drawer_name}: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()