# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The goal is to explore and discover which predicate is missing, specifically related to the robot's location.

        # We assume the robot starts at some initial location, and we want to check if the predicate (robot-at drawer-area) is missing.
        # We'll attempt to move the robot to 'drawer-area' using the available skills and observe the result.

        # Find candidate locations from positions
        location_keys = [k for k in positions.keys() if 'area' in k or 'room' in k or 'location' in k]
        if 'drawer-area' in positions:
            drawer_area = 'drawer-area'
        elif len(location_keys) > 0:
            drawer_area = location_keys[0]
        else:
            print("[Exploration] No suitable location found for drawer-area.")
            drawer_area = None

        # Find robot's current location
        robot_location = None
        for k in positions:
            if 'robot' in k and 'location' in positions[k]:
                robot_location = positions[k]['location']
                break
        if robot_location is None:
            # Fallback: try to find a location key that is not drawer-area
            for loc in location_keys:
                if loc != drawer_area:
                    robot_location = loc
                    break

        if drawer_area is None or robot_location is None:
            print("[Exploration] Could not determine robot or drawer-area location. Exploration aborted.")
            return

        print(f"[Exploration] Robot initial location: {robot_location}")
        print(f"[Exploration] Target location (drawer-area): {drawer_area}")

        # Try to move the robot to drawer-area using the available skill
        # The available skill for moving is 'execute_go'
        try:
            print(f"[Exploration] Attempting to move robot from {robot_location} to {drawer_area} using execute_go.")
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print(f"[Exploration] execute_go completed. Checking if robot is at {drawer_area}...")

            # After moving, check if the predicate (robot-at drawer-area) is now true in the environment
            # This may require querying the environment or checking the observation
            # For demonstration, we print the observation or state
            if hasattr(task, 'get_observation'):
                obs = task.get_observation()
                print(f"[Exploration] Observation after move: {obs}")
            else:
                print("[Exploration] No get_observation method available.")

            # If the predicate is still missing, this suggests the domain or state representation is missing (robot-at drawer-area)
            # This is the missing predicate to be reported/fixed.

        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")

        # === End of Exploration Phase ===

        # (Optional) Continue with further plan steps if needed, using only predefined skills.
        # For this task, the focus is on exploration and predicate discovery.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()