# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at drawer-area)
        # The plan failed because the predicate (robot-at drawer-area) was missing or not properly set.
        # We need to ensure the robot is at the correct location before executing further actions.

        # Let's try to move the robot to 'drawer-area' using the available skill 'execute_go'
        # and check if the predicate is set correctly.

        # Assume positions contains keys for locations, e.g., 'start-area', 'drawer-area'
        # If not, handle gracefully.

        # Find current robot location and drawer-area
        robot_location = None
        drawer_area = None

        # Try to infer location names from positions
        for key in positions:
            if 'drawer' in key and 'area' in key:
                drawer_area = key
            if 'robot' in key or 'start' in key:
                robot_location = key

        # Fallbacks if not found
        if drawer_area is None:
            # Try common names
            if 'drawer-area' in positions:
                drawer_area = 'drawer-area'
            else:
                # Pick any location with 'drawer' in name
                for k in positions:
                    if 'drawer' in k:
                        drawer_area = k
                        break
        if robot_location is None:
            # Try 'robot_start' or similar
            for k in positions:
                if 'start' in k:
                    robot_location = k
                    break
            # If still not found, just pick the first location
            if robot_location is None and len(positions) > 0:
                robot_location = list(positions.keys())[0]

        print(f"[Exploration] Robot location: {robot_location}, Drawer area: {drawer_area}")

        # Try to move robot to drawer-area using execute_go
        try:
            obs, reward, done = execute_go(
                env,
                task,
                from_location=robot_location,
                to_location=drawer_area,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Executed execute_go to drawer-area.")
        except Exception as e:
            print(f"[Exploration] Failed to execute_go: {e}")

        # After moving, check if the predicate (robot-at drawer-area) is now true
        # (Assume task.get_observation() returns a dict of predicates)
        try:
            current_obs = task.get_observation()
            if isinstance(current_obs, dict):
                if ('robot-at', drawer_area) in current_obs or \
                   ('robot-at', drawer_area.replace('_', '-')) in current_obs:
                    print("[Exploration] Predicate (robot-at drawer-area) is now true.")
                else:
                    print("[Exploration] Predicate (robot-at drawer-area) is still missing.")
            else:
                print("[Exploration] Observation format not recognized.")
        except Exception as e:
            print(f"[Exploration] Could not check predicate: {e}")

        # === Main Task Plan Execution ===
        # Now that the missing predicate is handled, proceed with the oracle plan.
        # Example: pick an object from the floor and place it in the drawer.

        # Find an object on the floor and the drawer
        object_on_floor = None
        for obj in positions:
            if 'object' in obj or 'ball' in obj or 'item' in obj:
                object_on_floor = obj
                break

        # Find a drawer object
        drawer_obj = None
        for obj in positions:
            if 'drawer' in obj and obj != drawer_area:
                drawer_obj = obj
                break

        # Find a handle object (if needed for pull)
        handle_obj = None
        for obj in positions:
            if 'handle' in obj:
                handle_obj = obj
                break

        # 1. Ensure room is bright (if not, push switch)
        # Try to execute_push_switch if available
        try:
            obs = task.get_observation()
            if isinstance(obs, dict):
                if 'room-dark' in obs and obs['room-dark']:
                    # Find a switch and its location
                    switch_obj = None
                    switch_loc = None
                    for obj in positions:
                        if 'switch' in obj:
                            switch_obj = obj
                            switch_loc = obj  # Assume switch is at its own location
                            break
                    if switch_obj and switch_loc:
                        obs, reward, done = execute_push_switch(
                            env,
                            task,
                            switch=switch_obj,
                            location=switch_loc,
                            max_steps=100,
                            threshold=0.01,
                            timeout=10.0
                        )
                        print("[Task] Executed execute_push_switch to turn on the light.")
        except Exception as e:
            print(f"[Task] Could not execute_push_switch: {e}")

        # 2. Pick up the object from the floor
        try:
            if object_on_floor and drawer_area:
                obs, reward, done = execute_pick(
                    env,
                    task,
                    obj=object_on_floor,
                    location=drawer_area,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Picked up {object_on_floor} at {drawer_area}.")
        except Exception as e:
            print(f"[Task] Could not execute_pick: {e}")

        # 3. Open the drawer (if needed)
        try:
            if drawer_obj and handle_obj and drawer_area:
                obs, reward, done = execute_pull(
                    env,
                    task,
                    drawer=drawer_obj,
                    handle=handle_obj,
                    location=drawer_area,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Pulled open {drawer_obj} using {handle_obj} at {drawer_area}.")
        except Exception as e:
            print(f"[Task] Could not execute_pull: {e}")

        # 4. Place the object in the drawer
        try:
            if object_on_floor and drawer_obj and drawer_area:
                obs, reward, done = execute_place(
                    env,
                    task,
                    obj=object_on_floor,
                    drawer=drawer_obj,
                    location=drawer_area,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Placed {object_on_floor} in {drawer_obj} at {drawer_area}.")
        except Exception as e:
            print(f"[Task] Could not execute_place: {e}")

        # 5. Close the drawer (optional)
        try:
            if drawer_obj and drawer_area:
                obs, reward, done = execute_push(
                    env,
                    task,
                    drawer=drawer_obj,
                    location=drawer_area,
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                print(f"[Task] Closed {drawer_obj} at {drawer_area}.")
        except Exception as e:
            print(f"[Task] Could not execute_push: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()