# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to execute all available skills in a safe exploratory way,
        # and observe which predicates are required or missing.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to execute each skill with plausible arguments, catching exceptions
        # and logging any issues that may indicate missing predicates.

        # For demonstration, we will attempt each skill with dummy or inferred arguments.
        # In a real scenario, you would use actual object/location names from the environment.

        # Get all object and location names from positions
        object_names = []
        location_names = []
        drawer_names = []
        handle_names = []
        switch_names = []

        for name, pos in positions.items():
            if 'drawer' in name:
                drawer_names.append(name)
            elif 'handle' in name:
                handle_names.append(name)
            elif 'switch' in name:
                switch_names.append(name)
            elif 'loc' in name or 'room' in name or 'area' in name:
                location_names.append(name)
            else:
                object_names.append(name)

        # Fallbacks if not found
        if not location_names:
            location_names = ['location1', 'location2']
        if not object_names:
            object_names = ['object1', 'object2']
        if not drawer_names:
            drawer_names = ['drawer1']
        if not handle_names:
            handle_names = ['handle1']
        if not switch_names:
            switch_names = ['switch1']

        # Use the first available location as the robot's starting point
        robot_location = location_names[0]
        other_location = location_names[1] if len(location_names) > 1 else location_names[0]

        # Exploration log
        exploration_results = {}

        # 1. Try execute_go (move robot)
        try:
            print("[Exploration] Trying execute_go from", robot_location, "to", other_location)
            obs, reward, done = execute_go(env, task, robot_location, other_location)
            exploration_results['execute_go'] = 'success'
        except Exception as e:
            print("[Exploration] execute_go failed:", e)
            exploration_results['execute_go'] = str(e)

        # 2. Try execute_pick (pick up an object)
        try:
            print("[Exploration] Trying execute_pick on", object_names[0], "at", other_location)
            obs, reward, done = execute_pick(env, task, object_names[0], other_location)
            exploration_results['execute_pick'] = 'success'
        except Exception as e:
            print("[Exploration] execute_pick failed:", e)
            exploration_results['execute_pick'] = str(e)

        # 3. Try execute_place (place object in drawer)
        try:
            print("[Exploration] Trying execute_place", object_names[0], "in", drawer_names[0], "at", other_location)
            obs, reward, done = execute_place(env, task, object_names[0], drawer_names[0], other_location)
            exploration_results['execute_place'] = 'success'
        except Exception as e:
            print("[Exploration] execute_place failed:", e)
            exploration_results['execute_place'] = str(e)

        # 4. Try execute_push (close drawer)
        try:
            print("[Exploration] Trying execute_push on", drawer_names[0], "at", other_location)
            obs, reward, done = execute_push(env, task, drawer_names[0], other_location)
            exploration_results['execute_push'] = 'success'
        except Exception as e:
            print("[Exploration] execute_push failed:", e)
            exploration_results['execute_push'] = str(e)

        # 5. Try execute_pull (open drawer with handle)
        try:
            print("[Exploration] Trying execute_pull on", drawer_names[0], "with", handle_names[0], "at", other_location)
            obs, reward, done = execute_pull(env, task, drawer_names[0], handle_names[0], other_location)
            exploration_results['execute_pull'] = 'success'
        except Exception as e:
            print("[Exploration] execute_pull failed:", e)
            exploration_results['execute_pull'] = str(e)

        # 6. Try execute_sweep (sweep object)
        try:
            print("[Exploration] Trying execute_sweep on", object_names[0], "at", other_location)
            obs, reward, done = execute_sweep(env, task, object_names[0], other_location)
            exploration_results['execute_sweep'] = 'success'
        except Exception as e:
            print("[Exploration] execute_sweep failed:", e)
            exploration_results['execute_sweep'] = str(e)

        # 7. Try execute_gripper (test gripper)
        try:
            print("[Exploration] Trying execute_gripper")
            obs, reward, done = execute_gripper(env, task)
            exploration_results['execute_gripper'] = 'success'
        except Exception as e:
            print("[Exploration] execute_gripper failed:", e)
            exploration_results['execute_gripper'] = str(e)

        # 8. Try execute_push_switch (turn on light if available)
        if 'execute_push_switch' in globals():
            try:
                print("[Exploration] Trying execute_push_switch on", switch_names[0], "at", robot_location)
                obs, reward, done = execute_push_switch(env, task, switch_names[0], robot_location)
                exploration_results['execute_push_switch'] = 'success'
            except Exception as e:
                print("[Exploration] execute_push_switch failed:", e)
                exploration_results['execute_push_switch'] = str(e)

        print("\n[Exploration Results]")
        for skill, result in exploration_results.items():
            print(f"  {skill}: {result}")

        print("\n[Exploration] If any skill failed due to a missing predicate, check the error messages above.")
        print("[Exploration] Use this information to update your domain/problem PDDL or to report the missing predicate.")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()