# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will use available skills to explore the environment and try to discover which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to interact with objects and observe the environment.

        # List of available skills
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper'
        ]

        # Try to interact with all objects and drawers in the environment
        # For demonstration, we will attempt to:
        # 1. Move to each location (execute_go)
        # 2. Pick up each object (execute_pick)
        # 3. Pull handles (execute_pull)
        # 4. Place objects (execute_place)
        # 5. Push drawers (execute_push)
        # 6. Sweep (execute_sweep)
        # 7. Use gripper (execute_gripper)
        # 8. Rotate (execute_rotate) if available

        # The actual object and location names should be obtained from the observation or positions
        # For this generic code, we will use placeholder names and handle missing keys gracefully

        # Example: positions = {'object_1': (x, y, z), 'drawer_1': (x, y, z), ...}
        object_names = [name for name in positions.keys() if 'object' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'room' in name]

        # If no explicit locations, use all unique positions as locations
        if not location_names:
            location_names = list(set([str(pos) for pos in positions.values()]))

        # 1. Try to move to each location
        for i, from_loc in enumerate(location_names):
            for to_loc in location_names:
                if from_loc == to_loc:
                    continue
                try:
                    print(f"[Exploration] Trying to move from {from_loc} to {to_loc} using execute_go")
                    obs, reward, done = execute_go(env, task, from_loc, to_loc)
                except Exception as e:
                    print(f"[Exploration] execute_go failed: {e}")

        # 2. Try to pick up each object at each location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pick {obj} at {loc} using execute_pick")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_pick failed: {e}")

        # 3. Try to pull each drawer handle (if handles are objects)
        for drawer in drawer_names:
            # Try to find a handle object for this drawer
            handle_obj = None
            for obj in object_names:
                if 'handle' in obj and drawer in obj:
                    handle_obj = obj
                    break
            if handle_obj is None and object_names:
                handle_obj = object_names[0]  # fallback to any object
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to pull {drawer} with handle {handle_obj} at {loc} using execute_pull")
                    obs, reward, done = execute_pull(env, task, drawer, handle_obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_pull failed: {e}")

        # 4. Try to place each object in each drawer at each location
        for obj in object_names:
            for drawer in drawer_names:
                for loc in location_names:
                    try:
                        print(f"[Exploration] Trying to place {obj} in {drawer} at {loc} using execute_place")
                        obs, reward, done = execute_place(env, task, obj, drawer, loc)
                    except Exception as e:
                        print(f"[Exploration] execute_place failed: {e}")

        # 5. Try to push each drawer at each location
        for drawer in drawer_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to push {drawer} at {loc} using execute_push")
                    obs, reward, done = execute_push(env, task, drawer, loc)
                except Exception as e:
                    print(f"[Exploration] execute_push failed: {e}")

        # 6. Try to sweep each object at each location
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Trying to sweep {obj} at {loc} using execute_sweep")
                    obs, reward, done = execute_sweep(env, task, obj, loc)
                except Exception as e:
                    print(f"[Exploration] execute_sweep failed: {e}")

        # 7. Try to use gripper (if skill takes no arguments)
        try:
            print(f"[Exploration] Trying to use gripper using execute_gripper")
            obs, reward, done = execute_gripper(env, task)
        except Exception as e:
            print(f"[Exploration] execute_gripper failed: {e}")

        # 8. Try to rotate (if available)
        if 'execute_rotate' in available_skills:
            try:
                print(f"[Exploration] Trying to rotate using execute_rotate")
                obs, reward, done = execute_rotate(env, task)
            except Exception as e:
                print(f"[Exploration] execute_rotate failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()