# run_skeleton_task.py (Completed for Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The feedback indicates a missing predicate is blocking planning.
        # We will attempt to use available skills to explore the environment and infer missing predicates.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will try to use the available skills to interact with objects/drawers and observe the environment's response.

        # List of available skills (from provided list)
        available_skills = [
            'execute_pick', 'execute_place', 'execute_push', 'execute_pull',
            'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper',
            'execute_push_swtich -> push'
        ]

        # Try to get all objects, drawers, locations, switches from the environment
        # This is a generic approach; adapt as needed for your environment
        objects = []
        drawers = []
        locations = []
        switches = []
        handles = []

        # Parse positions dict to categorize objects
        for name in positions:
            lname = name.lower()
            if 'drawer' in lname:
                drawers.append(name)
            elif 'switch' in lname:
                switches.append(name)
            elif 'handle' in lname:
                handles.append(name)
            elif 'loc' in lname or 'room' in lname or 'area' in lname:
                locations.append(name)
            else:
                objects.append(name)

        # If locations are not found, try to infer from robot position or object positions
        if not locations:
            # Fallback: use unique locations from object positions
            loc_set = set()
            for pos in positions.values():
                if isinstance(pos, dict) and 'location' in pos:
                    loc_set.add(pos['location'])
            locations = list(loc_set)
        if not locations:
            # As a last resort, use a default location
            locations = ['location_1']

        # Try to get robot's initial location
        robot_location = None
        for name, pos in positions.items():
            if 'robot' in name.lower() and isinstance(pos, dict) and 'location' in pos:
                robot_location = pos['location']
                break
        if robot_location is None and locations:
            robot_location = locations[0]

        # === Exploration: Try available skills to trigger environment feedback ===
        # The goal is to interact with objects/drawers to see if any action fails due to a missing predicate.
        # We will attempt to pick, pull, push, and place objects/drawers and observe any errors or feedback.

        exploration_done = False
        for obj in objects:
            for loc in locations:
                # Try to pick the object (if possible)
                try:
                    print(f"[Exploration] Trying execute_pick on {obj} at {loc}")
                    obs, reward, done = execute_pick(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Successfully picked {obj} at {loc}")
                        exploration_done = True
                        break
                except Exception as e:
                    print(f"[Exploration] execute_pick failed for {obj} at {loc}: {e}")

            if exploration_done:
                break

        # Try to interact with drawers (pull, push, place)
        if not exploration_done:
            for drawer in drawers:
                for handle in handles:
                    for loc in locations:
                        # Try to pull the drawer using its handle
                        try:
                            print(f"[Exploration] Trying execute_pull on {drawer} with handle {handle} at {loc}")
                            obs, reward, done = execute_pull(env, task, drawer, handle, loc)
                            if done:
                                print(f"[Exploration] Successfully pulled {drawer} with {handle} at {loc}")
                                exploration_done = True
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_pull failed for {drawer} with {handle} at {loc}: {e}")
                    if exploration_done:
                        break
                if exploration_done:
                    break

        # Try to push drawers (close them)
        if not exploration_done:
            for drawer in drawers:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_push on {drawer} at {loc}")
                        obs, reward, done = execute_push(env, task, drawer, loc)
                        if done:
                            print(f"[Exploration] Successfully pushed {drawer} at {loc}")
                            exploration_done = True
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_push failed for {drawer} at {loc}: {e}")
                if exploration_done:
                    break

        # Try to place objects in drawers
        if not exploration_done:
            for obj in objects:
                for drawer in drawers:
                    for loc in locations:
                        try:
                            print(f"[Exploration] Trying execute_place of {obj} in {drawer} at {loc}")
                            obs, reward, done = execute_place(env, task, obj, drawer, loc)
                            if done:
                                print(f"[Exploration] Successfully placed {obj} in {drawer} at {loc}")
                                exploration_done = True
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_place failed for {obj} in {drawer} at {loc}: {e}")
                    if exploration_done:
                        break
                if exploration_done:
                    break

        # Try to sweep objects
        if not exploration_done:
            for obj in objects:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_sweep on {obj} at {loc}")
                        obs, reward, done = execute_sweep(env, task, obj, loc)
                        if done:
                            print(f"[Exploration] Successfully swept {obj} at {loc}")
                            exploration_done = True
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_sweep failed for {obj} at {loc}: {e}")
                if exploration_done:
                    break

        # Try to turn on the light if the room is dark (execute_push_switch)
        if not exploration_done and switches:
            for switch in switches:
                for loc in locations:
                    try:
                        print(f"[Exploration] Trying execute_push_switch on {switch} at {loc}")
                        obs, reward, done = execute_push_switch(env, task, switch, loc)
                        if done:
                            print(f"[Exploration] Successfully pushed switch {switch} at {loc}")
                            exploration_done = True
                            break
                    except Exception as e:
                        print(f"[Exploration] execute_push_switch failed for {switch} at {loc}: {e}")
                if exploration_done:
                    break

        # Try to move between locations (execute_go)
        if not exploration_done and len(locations) > 1:
            for i in range(len(locations)):
                for j in range(len(locations)):
                    if i != j:
                        try:
                            print(f"[Exploration] Trying execute_go from {locations[i]} to {locations[j]}")
                            obs, reward, done = execute_go(env, task, locations[i], locations[j])
                            if done:
                                print(f"[Exploration] Successfully moved from {locations[i]} to {locations[j]}")
                                exploration_done = True
                                break
                        except Exception as e:
                            print(f"[Exploration] execute_go failed from {locations[i]} to {locations[j]}: {e}")
                if exploration_done:
                    break

        # Try to use gripper
        if not exploration_done:
            try:
                print(f"[Exploration] Trying execute_gripper")
                obs, reward, done = execute_gripper(env, task)
                if done:
                    print(f"[Exploration] Successfully used gripper")
                    exploration_done = True
            except Exception as e:
                print(f"[Exploration] execute_gripper failed: {e}")

        # If all exploration attempts failed, report
        if not exploration_done:
            print("[Exploration] All exploration attempts failed. The missing predicate may require a new skill or observation.")

        # === End of Exploration Phase ===

        # At this point, you should have observed which action(s) failed and what predicate(s) might be missing.
        # Use this information to update your PDDL or planning logic as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()