# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (robot-at loc_a)
        # The goal is to explore and determine which predicate is missing, using only available skills.

        # Available skills:
        # ['execute_pick', 'execute_place', 'execute_push', 'execute_pull', 'execute_sweep', 'execute_rotate', 'execute_go', 'execute_gripper']

        # We will attempt to use 'execute_go' to move the robot and check if the predicate (robot-at ...) is updated.
        # If not, we infer that the predicate is missing or not being set as expected.

        # For demonstration, let's assume there are at least two locations in the environment.
        # We'll try to move the robot from one location to another using 'execute_go'.

        # Find two distinct locations from the positions dictionary
        location_names = [name for name in positions if name.startswith('loc_') or name.startswith('location') or 'loc' in name]
        if len(location_names) < 2:
            # Fallback: try to find any two unique locations
            location_names = list(set([v['location'] if isinstance(v, dict) and 'location' in v else k for k, v in positions.items()]))
        if len(location_names) < 2:
            print("[Exploration] Not enough locations found for exploration.")
            return

        from_location = location_names[0]
        to_location = location_names[1]

        print(f"[Exploration] Attempting to move robot from {from_location} to {to_location} using execute_go.")

        try:
            # Call the predefined skill to move the robot
            obs, reward, done = execute_go(
                env,
                task,
                from_location,
                to_location
            )
            print(f"[Exploration] Called execute_go({from_location}, {to_location}).")
        except Exception as e:
            print(f"[Exploration] Exception during execute_go: {e}")
            return

        # After moving, check if the robot's location predicate is updated
        # This would typically be in the observation or state
        # For demonstration, we print the observation and look for 'robot-at' or similar
        print("[Exploration] Observation after execute_go:")
        print(obs)

        # Try to check if the predicate (robot-at to_location) is present in the observation
        robot_at_found = False
        if isinstance(obs, dict):
            for k, v in obs.items():
                if 'robot-at' in k and to_location in str(v):
                    robot_at_found = True
                    break
        elif isinstance(obs, (list, tuple)):
            for item in obs:
                if isinstance(item, str) and 'robot-at' in item and to_location in item:
                    robot_at_found = True
                    break
        else:
            obs_str = str(obs)
            if 'robot-at' in obs_str and to_location in obs_str:
                robot_at_found = True

        if robot_at_found:
            print(f"[Exploration] Predicate (robot-at {to_location}) found in observation. Predicate is present.")
        else:
            print(f"[Exploration] Predicate (robot-at {to_location}) NOT found in observation. Predicate may be missing.")

        # === End of Exploration Phase ===

        # You can continue with further exploration or task logic as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
